import logging
from functools import wraps
from time import sleep

import grpc

log = logging.getLogger(__name__)


def grpc_retry(max_attempts: int = 5, start_delay: float = 1.0):
    def decorator(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):
            for attempt_number in range(1, max_attempts + 1):
                try:
                    return func(self, *args, **kwargs)
                except grpc.RpcError as e:
                    if e.code() in (grpc.StatusCode.UNAVAILABLE, grpc.StatusCode.INTERNAL):
                        if attempt_number == max_attempts:
                            log.warning("Exceeded command retry attempts")
                            raise e
                        timeout = start_delay * 2 ** attempt_number
                        log.warning(f"Got error: {e}, wait for {timeout} seconds and retry the command")
                        sleep(timeout)
                    else:
                        raise e

        return wrapper

    return decorator
