import csv
import json

import pandas as pd

from tqdm import tqdm
from datetime import datetime
from assets import Asset


class EnerkeyReader:
    def __init__(self, ids_file='data/enerkey-ids.csv', hourly_data_file='data/enerkey_hourly.json', bucket=None,
                 firebase=None):
        self._ids_file = ids_file
        self._hourly_data_file = hourly_data_file
        self._map = self._map_enerkey_ids(self._ids_file)
        self._rawdata = self._load_data(self._hourly_data_file)
        self.assets_list, self.assets = self._sort_data_in_assets(self._rawdata, self._map)
        self._bucket = bucket
        self._firebase = firebase

    def _map_enerkey_ids(self, filename):

        with open(filename, 'r') as f:
            csvfile=csv.reader(f)
            header = next(csvfile)
            rows = []
            for row in csvfile:
                rows.append(row)

            map = {header[0]: {}}
            for row in rows:
                map[header[0]].update({row[0]: {header[1]: row[1], header[2]: row[2]}})
            return map

    def _load_data(self, filename):
        with open(filename, 'r') as f:
            data=json.load(f)

        return data

    def _sort_data_in_assets(self, data, map):
        assets = {}
        assets_list = []
        for v in tqdm(data.values()):
            nrep_id = map['Enerkey_Name'][v['buildingName']]['NREP_XRM_Asset_ID']
            values = [vv.get('value') for vv in v['values']]
            times = [datetime.fromisoformat(vv.get('timestamp')).timestamp() for vv in v['values']]
            if nrep_id not in assets:
                assets[nrep_id] = Asset(id=nrep_id)
            assets[nrep_id].add_meter(name=v['measurementName'], meterId=v['meterId'], enegiaId=v['enegiaId'],
                                          unit=v['unitName'], df=pd.DataFrame({'times': times, 'values': values}),
                                          start=times[0], end=times[-1])

        for asset in assets.values():
            assets_list.append(asset)
        return assets_list, assets

    def save_assets(self):
        for asset in tqdm(self.assets_list):
            asset.save("../outputs")

if __name__ == "__main__":

    e = EnerkeyReader()
    e.save_assets()
