import json
import logging
import os

import uuid
from importlib.metadata import version
from paho.mqtt import client as mqtt
from threading import Event
from time import sleep, time

from numerous.image_tools.app import NumerousApp
from numerous.image_tools.job import NumerousBaseJob

from numerous.client import NumerousClient

logger = logging.getLogger('mqtt_runner')

VERSION = version('numerous_image_tools')


class NumerousMQTTApp(NumerousApp):
    def __init__(self, appname="defaultnumerousApp",
                 max_restarts=0, numerous_job: NumerousBaseJob = None, model_folder: str = None,
                 working_folder: str = None, reset_job=False, mqtt_client: mqtt.Client = None, project_id: str = None,
                 simulation_id: str = None):
        super(NumerousMQTTApp, self).__init__(appname=appname,
                                              max_restarts=max_restarts, numerous_job=numerous_job,
                                              model_folder=model_folder,
                                              working_folder=working_folder, reset_job=reset_job)

        assert simulation_id, "No simulation id set"
        assert project_id, "No em cloud project id set"

        hostname = self.client.params.get('MQTT_HOSTNAME', os.getenv('MQTT_HOSTNAME'))
        password = self.client.params.get('MQTT_PASSWORD', os.getenv('MQTT_PASSWORD'))
        port = self.client.params.get('MQTT_PORT', os.getenv('MQTT_PORT'))
        username = self.client.params.get('MQTT_USERNAME', os.getenv('MQTT_USERNAME'))
        logger.debug(hostname)
        logger.debug(port)
        logger.debug(username)

        self._project_id = project_id
        self._simulation_id = simulation_id
        self.mqtt_client = mqtt_client
        self.mqtt_client.on_connect = self.on_connect
        self.mqtt_client.on_message = self.on_message
        self.mqtt_client.username_pw_set(username=username, password=password)
        self.mqtt_client.connect(host=hostname, port=int(port))

        self._insert_response = f"{self._project_id}/simulation/insertResponse"
        self._delete_response = f"{self._project_id}/simulation/deleteResponse"
        self._delete_request = f"{self._project_id}/simulation/deleteRequest"
        self._delete_id = None
        self._cleared = Event()

        class __Output:

            def __init__(self, numerous_client: NumerousClient, mqtt_client: mqtt.Client, project_id: str,
                         simulation_id: str):
                self._numerous_output = numerous_client.new_writer(buffer_size=0)
                self._mqtt_client = mqtt_client
                self._project_id = project_id
                self._simulation_id = simulation_id
                self.cnt = 0
                self._ids = set()
                self._insert_request = f"{self._project_id}/simulation/insertRequest"

            def write_row(self, t, row):
                self._numerous_output.write_row(row)
                msg = self._mqtt_client.publish(self._insert_request, self.format_scheme(t, row))

            def format_scheme(self, t: float, row: dict):
                _id = uuid.uuid4()
                self._ids.add(str(_id))
                outputs = []
                for tag, val in row.items():
                    if tag == "_index":
                        continue
                    outputs.append({
                        "path": f"{self._simulation_id}/{tag}",
                        "type": "float",
                        "unit": "",
                        "times": [int(t*1000)],  # seconds to milliseconds
                        "values": [val]
                    })

                request = {'id': str(_id),
                           'data': outputs}
                return json.dumps(request)

            def close(self):
                self._numerous_output.close()
                self._mqtt_client.loop_stop()

        self.output = __Output(self.client, self.mqtt_client, self._project_id, self._simulation_id)
        self.mqtt_client.loop_start()
        sleep(1)
        states = self.client.state.get('states', None)

        if reset_job and not states:
            self.delete_simulation()

    def on_connect(self, client: mqtt.Client, userdata, flags, rc):
        logger.debug(f"client connected: {rc}")
        client.subscribe(self._insert_response)
        client.subscribe(self._delete_response)

    def on_message(self, client: mqtt.Client, userdata, msg):
        logger.debug(f"message on topic {msg.topic}: {msg.payload}")
        _msg = json.loads(msg.payload.decode())
        _id = _msg.get('id')
        status = _msg.get('success')
        if msg.topic == self._insert_response and _id in self.output._ids and status:
            self.output._ids.remove(_id)
        if msg.topic == self._delete_response and self._delete_id == _id and status:
            self._cleared.set()
            self._delete_id = None

    def _delete_simulation(self):
        self._delete_id = str(uuid.uuid4())
        body = {"id": self._delete_id,
                "path": f"{self._simulation_id}/"}
        self.mqtt_client.publish(self._delete_request,json.dumps(body))

    def delete_simulation(self):
        self._delete_simulation()
        t0 = time()
        tmax = 60
        while not self._cleared.is_set():
            sleep(0.1)
            if time() - t0 > tmax:
                logger.error(f"simulation data not cleared after {tmax} seconds")
                return False
        logger.info("simulation data cleared")
        return True


def run_mqtt_job(numerous_job=None, appname=None, max_restarts=0, model_folder="models", working_folder="tmp",
                 reset_job=None, project_id=None, simulation_id=None):
    app = NumerousMQTTApp(appname=appname,
                          max_restarts=max_restarts, numerous_job=numerous_job, model_folder=model_folder,
                          working_folder=working_folder, reset_job=reset_job, project_id=project_id,
                          simulation_id=simulation_id, mqtt_client=mqtt.Client())

    app._run()
