import queue
import math
import numpy as np
import pandas as pd
import logging

from queue import Empty

from .numerous_system_initializer import SystemInitializer

from numerous.engine.simulation.simulation import Simulation
from numerous.engine.simulation.solvers import SolverType
from numerous.engine.model.model import Model
from numerous.engine.system.external_mappings.interpolation_type import InterpolationType
from numerous.engine.system.external_mappings import ExternalMappingElement

from numerous.utils.data_loader import DataLoader, DataFrame
from numerous.utils.logger_levels import LoggerLevel
from numerous.utils.historian import Historian
from numerous.image_tools.job import NumerousSimulationJob
from numerous.image_tools.system import NumerousSystem

logger = logging.getLogger(__file__)


try:
    FEPS = np.finfo(1.0).eps
except AttributeError:
    FEPS = 2.220446049250313e-16


def map_to_dataframe(system: NumerousSystem, t: float, index_to_timestep_mapping: str):
    df_dict = {}
    for component in system.components.values():
        for alias, column_name in component.parameters.items():
            df_dict.update({column_name: [component.inputs.get(alias, None)]})

    df_dict.update({index_to_timestep_mapping: [t]})

    return pd.DataFrame(df_dict, dtype=np.float64)


class NumerousEngineJob(NumerousSimulationJob):
    def __init__(self):
        super().__init__()
        self.tag = 'system'
        self.len_input_data = None
        self.df_local_path = None
        self.simulation = None
        self.dt = None
        self.t_simulation = None
        self.t_stop = None
        self.t_start = None
        self.t_offset = None
        self.y0 = None
        self._metadata_written = False

        self.dataframe_aliases = {}
        self.external_mappings = []

        class __NumerousDataLoader(DataLoader):

            def __init__(self, job: NumerousEngineJob):
                super().__init__()
                self.job = job
                self.df = pd.DataFrame()


            def load(self, df_id: str, t: int) -> DataFrame:
                n = math.floor(t / self.job.dt + FEPS * 100) + 1
                t_eval = np.linspace(t, n * self.job.dt, 2)
                t_end = t_eval[-1]

                next_val = self.job.read_data(self.job.t_offset + t_end)
                self.job.system.update_inputs(next_val)
                df_ = map_to_dataframe(self.job.system, t_end, 't')

                df = pd.concat([self.df, df_], axis=0)
                self.df = df[len(df)-2:]  # Drop previous timestep

                return self.df

        class __NumerousHistorian(Historian):
            def __init__(self, job: NumerousEngineJob, max_size=None):
                super().__init__(max_size)
                self.max_size = max_size
                self._historians = []
                self.df = pd.DataFrame()
                self.job = job
                self._queue = queue.Queue()

            def store(self, df):
                self.add_dataframe(df)
                self.df = df

            def add_dataframe(self, df):
                output_dict = {col: df[col].values for col in self.job.simulation.model.logged_variables}
                output_dict.update({"_index": df['time'].values})  # best practise is to save relative time

                outputs = [dict(zip(output_dict, t)) for t in zip(*output_dict.values())]
                for output in outputs:
                    self._queue.put(output)

            def get(self):
                try:
                    return self._queue.get(block=False)
                except Empty:
                    return

        self.data_loader = __NumerousDataLoader(self)
        self.historian = __NumerousHistorian(self, max_size=100)

    def setup_mappings(self, start: float, time_multiplier: float, index_to_timestep_mapping: str,
                       index_to_timestep_mapping_start: int):
        self.dataframe_aliases = {}

        for component in self.system.components.values():
            name = component.name

            for alias, column_name in component.parameters.items():
                mapped_variable = f"{self.tag}.{name}.{alias}"
                self.dataframe_aliases.update({mapped_variable: (column_name, InterpolationType.PIESEWISE)})

        df = map_to_dataframe(self.system, start, index_to_timestep_mapping)

        # Add external mappings so data can be read
        eme = ExternalMappingElement(
            "n/a",
            index_to_timestep_mapping,
            index_to_timestep_mapping_start,
            time_multiplier,
            self.dataframe_aliases
        )
        eme.add_df(df)
        self.data_loader.df = df

        self.external_mappings = [eme]

    def initialize_simulation_system(self):

        self.dt= self.system.dt
        self.t_offset = self.system.start_time

        self.y0 = self.system.states
        self.t_start = self.system.states.get('t', self.system.start_time)-self.system.start_time
        self.len_input_data = 1

        self.t_simulation = self.system.end_time - self.system.start_time - self.t_start
        self.t_stop = self.t_start + self.dt

        self.setup_mappings(self.t_start, 1, 't', 0)

        # Create a system based on specified components
        enginesystem = SystemInitializer(
            self.tag,
            system=self.system, external_mappings=self.external_mappings if self.external_mappings else None,
            data_loader=self.data_loader if self.external_mappings else None
        )

        self.enginesystem = enginesystem

        # Create model based on system
        """
        model = Model(
            enginesystem, external_mappings=enginesystem.external_mappings,
            data_loader=LineByLineDataLoader(enginesystem.external_mappings), historian=self.historian,
            logger_level=LoggerLevel.INFO, imports=[("external_functions", "if_replacement_1"),
                                                              ("external_functions", "if_replacement_11"),
                                                              ("external_functions", "if_replacement_12"),
                                                              ("external_functions", "if_replacement_1_1")],
            use_llvm=True
        )
        """
        model = Model(
            enginesystem, historian=self.historian,
            logger_level=LoggerLevel.INFO,
            use_llvm=True, **self.system.parameters
        )

        # Create simulation object
        #simulation = None
        simulation = Simulation(
            model, t_start=self.t_start, t_stop=self.t_stop,
            num=1, num_inner=1,
            max_step=self.dt, solver_type=SolverType.NUMEROUS
        )

        if self.y0 is not None:
            for i, y_ in enumerate(self.y0):
                simulation.model.numba_model.write_variables(y_, i)

        self.simulation = simulation

        tags = []
        for name in model.logged_aliases.keys():
            tags.append({'name': name})

        self.init = False

    def read_input(self, t):
        """
        The logic of reading and passing external data is inside NumerousDataLoader. This function is therefore
        ignored.
        """
        return

    def write_output(self, t, output):
        output = self.historian.get()
        if not output:
            return

        if not self._metadata_written:
            self.app.client.set_timeseries_meta_data([{"name": tag} for tag in output.keys()],
                                                     offset=self.app.start_time)

        while output:
            self.app.output.write_row(t, output)
            output = self.historian.get()

    def serialize_states(self, t: float = None):
        states = list(self.simulation.model.numba_model.read_variables())
        return states

    def initial_step(self, t_: float, dt: float = None):
        _, tint = self.simulation.step_solve(t_, min(dt, self.t_simulation-t_))

    def step(self, t: float = None, dt: float = None, init=False):

        t_ = self.simulation.solver.info.t if self.simulation.solver.info else 0
        #self._update_input(t_)
        # watch out for round-off error!
        _, t_step = self.simulation.step_solve(t_, min(dt, self.t_simulation-t_))

        if t_step >= self.t_simulation:
            self.simulation.model.create_historian_df()

        return t_step + self.t_offset, None






