from numerous.engine.system import Subsystem, Item
from numerous.multiphysics import EquationBase, Equation
from numerous.image_tools.system import NumerousSystem
from numerous.engine.model import LoggerLevel

def entrypoint(tag: str, system: NumerousSystem):
    component = system.components[tag]

    model=NewtonsLawOfCooling(tag=tag, initial_temperature_of_hot_item=component.constants.get('temperature_of_hot_item',
                                                                                                 100),
                              initial_temperature_of_ambient=component.inputs.get('ambient.t1.t_ambient',0),
                              heat_capacity_of_hot_item=component.constants.get('heat_capacity_of_hot_item'),
                              heat_transfer_rate=component.constants.get('heat_transfer_rate'),
                              mass_of_hot_item=component.constants.get('mass_of_hot_item'))

    return model

class Hotitem(Item, EquationBase):
    def __init__(self, tag = 'decay', mass = None, heat_capacity = None, initial_temperature = None):
        super(Hotitem, self).__init__(tag)
        self.t1 = self.create_namespace('t1')
        self.add_constant('m', mass)
        self.add_constant('cv', heat_capacity)
        self.add_parameter('heatloss', 0)
        self.add_state('temperature', initial_temperature)
        self.t1.add_equations([self])

    @Equation()
    def mass(self, scope):
        scope.temperature_dot = scope.heatloss / (scope.m * scope.cv)


class Ambient(Item, EquationBase):
    def __init__(self, tag='ambient', initial_temperature = None):
        super(Ambient, self).__init__(tag)
        self.t1 = self.create_namespace('t1')
        self.add_parameter('temperature', initial_temperature)

        self.t1.add_equations([self])


class NewtonsLawOfCooling(Subsystem, EquationBase):
    def __init__(self, tag = "newtons_law_of_cooling", mass_of_hot_item = None, heat_capacity_of_hot_item = None,
                 initial_temperature_of_hot_item=None,
                 initial_temperature_of_ambient=None,
                 heat_transfer_rate = None):
        super(NewtonsLawOfCooling, self).__init__(tag)
        mass = Hotitem(mass=mass_of_hot_item, heat_capacity=heat_capacity_of_hot_item,
                    initial_temperature=initial_temperature_of_hot_item)
        ambient = Ambient(initial_temperature=initial_temperature_of_ambient)
        self.t1 = self.create_namespace('t1')
        self.add_parameter('t_mass', initial_temperature_of_hot_item, logger_level=LoggerLevel.INFO)
        self.add_parameter('t_ambient', initial_temperature_of_ambient)
        self.add_constant('h', heat_transfer_rate)
        self.add_parameter('q', 0)
        self.register_items([mass, ambient])
        self.t1.add_equations([self])
        self.t1.t_mass = mass.t1.temperature
        self.t1.t_ambient = ambient.t1.temperature
        mass.t1.heatloss += self.t1.q
    @Equation()
    def energy_transfer(self, scope):
        scope.q = -scope.h * (scope.t_mass-scope.t_ambient)

if __name__ == "__main__":
    from numerous.engine.simulation import Simulation
    from numerous.engine.model import Model
    import pandas as pd
    system = NewtonsLawOfCooling(tag="cooling", mass_of_hot_item=1, initial_temperature_of_hot_item=100,
                              initial_temperature_of_ambient=20,
                              heat_capacity_of_hot_item=1000,
                              heat_transfer_rate=100)
    model = Model(system=system, logger_level=LoggerLevel.INFO)

    sim = Simulation(model=model, t_start=0, t_stop=100, num=100, num_inner=1)
    sim.solve()
    df = pd.DataFrame(sim.model.create_historian_dict())

    with pd.ExcelWriter('test_newton' + ".xlsx") as writer:
        df.to_excel(writer, sheet_name='results')

    print(df)




