import copy

from numerous_api_client.client.numerous_client import ScenarioStatus
import os
import sys
import logging
import importlib
from pathlib import Path

logger = logging.getLogger('sim-setup')

class Component:
    def __init__(self, name: str = None, folder: str = None, path: str = None, handle: str = None,
                 constants: dict = None, parameters: dict = None,
                 import_flag = False):
        self._folder = Path(folder)
        self._path = Path(path)
        self._handle = handle
        self.constants = constants
        self.parameters = parameters
        self._constructor = None
        self._entrypoint = None
        self.item = None
        self.name = name
        self._import = import_flag
        self.inputs = {}
        self.model = None

    def _initialize(self, system):
        if not self._import:
            return
        item = self._entrypoint(self.name, system)
        self.item = item
        return item

    def _dynamicimport(self):
        if not self._import:
            return
        _entrypoint = getattr(importlib.import_module(self._path.as_posix()), self._handle)
        self._entrypoint = _entrypoint

class NumerousSystem:
    def __init__(self, nc=None, scenario=None, files=None, start_time=None, model_folder=None, numerous_job=None,
                 states=None, dt=None):

        self.nc = nc
        self.scenario = scenario
        self.files = files
        self.start_time = start_time

        # Update status - clear current data
        self.nc.set_scenario_progress('Environment Initializing',ScenarioStatus.ENVIRONMENT_INITIALIZING, 0.0, clean=True)

        # Initialize other system variables
        self.component_list, self.components = format_job_spec(self.scenario, model_folder)

        # Download needed files
        self._download_custom_files()
        self._dynamic_import_components()
        self.numerous_job = numerous_job
        self.states = states
        self.dt = dt
        self.parameters = {}
        for par in self.scenario.get('parameters', []):
            if 'id' in par and 'value' in par:
                self.parameters.update({par.get('id'): par.get('value')})

        logger.debug("completed setup")


    def _add_path_topdown(self, top):
        if top in sys.path:
            return
        for root,_, __ in os.walk(top, topdown=True):
            sys.path.append(root)

    def _download_custom_files(self):
        # Check if there are any custom files and download them if so
        datasets = self.scenario.get('datasets', [])
        if len(datasets) > 0:
            files = self.nc.get_download_files(datasets)

    def initialize_model(self):
        self.numerous_job.system = self

        for component in self.components.values():
            component.model = component._initialize(self)

        return self.numerous_job.initialize_simulation_system()

    def _dynamic_import_components(self):
        for component in self.components.values():
            self._add_path_topdown(component._folder)
            component._dynamicimport()

    def update_inputs(self, data):
        component_data = copy.deepcopy(data)
        component_data.pop('_index')
        component_data.pop('_index_relative')
        for input_tag, value in component_data.items():
            try:
                component_name = input_tag.split(".")[0]
                subcomponents = input_tag.split(".")[1:]
            except IndexError:
                logger.error(f'input {input_tag} could not be split into component and subcomponent')
                raise

            if component_name in self.components:
                self.components[component_name].inputs.update({".".join(subcomponents): value})

def format_job_spec(scenario, model_folder_name):

    simcomponents = scenario['simComponents']
    component_list = []
    components = {}

    for simcomponent in simcomponents:
        model_name = None
        model_file_name = None
        handle = None

        if not simcomponent['disabled']:
            # Get values for helper variables
            item_class = simcomponent['item_class']
            items = item_class.split(".")
            component_name = simcomponent['name']
            if item_class and len(items) < 3:
                raise IndexError(f"item {component_name} must have class of format x.y.z")

            if item_class:
                model_name = items[0]
                model_file_name = items[1]
                handle = items[2]

            # Format constant values ('parameters' in frontend)
            component_constants = {
                d['id']: d['value']
                for d in simcomponent['parameters']
            } if simcomponent['parameters'] else {}

            # Format parameters ('inputVariables' in frontend)
            component_parameters = {
                d['id']: f'{simcomponent["name"]}.{d["id"]}'
                for d in simcomponent['inputVariables']
            } if simcomponent['inputVariables'] else {}

            # Make a complete dict with formatted specifications

            component = Component(name=component_name,
                                  folder=f"{model_folder_name}/{model_name}",
                                  path=f"{model_folder_name}.{model_name}.{model_file_name}", handle=handle,
                                  constants=component_constants,
                                  parameters=component_parameters, import_flag=bool(item_class))
            component_list.append(component)
            components.update({component_name: component})

    # Return list of components

    return component_list, components