import os
import logging
from datetime import datetime
from time import sleep

from .errors import tb_str, SimulationError
from .system import NumerousSystem
from .report.report import Report
from numerous_api_client.client.data_source import DataSourceHibernating, DataSourceEmptyDataset, \
    DataSourceStreamClosed, DataSourceCompleted
from numerous_api_client.client.numerous_client import ScenarioStatus

LOG_LEVEL = os.getenv('LOG_LEVEL', logging.DEBUG)

class NumerousBaseJob:
    def __init__(self):
        self.system = None

    def serialize_states(self, t: float = None):
        """
        A method called when saving states.
        Returns: states as a json serializable object, to be saved.
        """
        return {}

    def _run_job(self, app):
        raise NotImplementedError


class NumerousReportJob(NumerousBaseJob):
    def __init__(self):
        super(NumerousReportJob, self).__init__()
        self.report = None
        self.template = f"{os.path.dirname(__file__)}/report/template/report_template_em.html"
        self.logger = logging.getLogger('numerous-report-job')
        self.logger.setLevel(level=LOG_LEVEL)

    def _run_job(self, app):
        try:
            self.report = Report(app.nc.upload_file, template=self.template)
            self.logger.info("adding report content")
            self.add_report_content(app)
            self.logger.info("finalizing report")
            self.report.finalize()
        except Exception as e:
            self.logger.error(f'report job crashed: {tb_str(e)}')
            app.status = -1
            app.message = "app error - see logs"
        finally:
            if app.status == 0:
                app.terminate.set()
            elif app.status == 1:
                app.terminate.set()
            elif app.status == 2:
                app.terminate.set()
            self.logger.warning('job terminated')

    def add_report_content(self, app):
        raise NotImplementedError

class NumerousSimulationJob(NumerousBaseJob):
    def __init__(self):
        super(NumerousSimulationJob, self).__init__()
        self.system = None
        self.align_outputs_to_next_timestep = True
        self.logger = logging.getLogger('numerous-simulation-job')
        self.logger.setLevel(level=LOG_LEVEL)

    def initialize_simulation_system(self):
        """
        A method that is called once the first data is read. All initialization should be done here.
        Can return the initial output to be saved.
        """
        return

    def step(self, t: float = None, dt: float = None):
        """
        A method that is called after each data read. Could be a step solver, or some other data manipulating function.
        Returns: tuple of next timestamp and outputs as a dict with tags to be saved

        """
        return t+dt, {"no_job_defined": None}

    def _run_job(self, app):

        # This is the classic pipelines approach - well suited for digital twins
        # Run simulation in a loop
        self.logger.debug(f'entering loop')
        timeout = 120
        cause = "no cause"
        details = ""
        i = 0
        last_data = None
        data = None
        t = None
        try:
            app.nc.set_scenario_progress('bootup', ScenarioStatus.INITIALZING, 0.0, force=True)
            app.terminate.wait(timeout=5)
            app.nc.set_scenario_progress('waiting for initial data', ScenarioStatus.INITIALZING, 0.0, force=True)

            self.output = app.nc.new_writer(buffer_size=0)
            t, states = app.load_states()
            t0 = t
            dt = app.nc.params.get('dt_simulation', 60)
            t_stop = app.end_time if not app.subscribe else 0

            self.system = NumerousSystem(app.nc, app.scenario, app.files, app.start_time,
                                         app.model_folder, self, states, dt)

            input = app.nc.get_inputs(app.scenario, t0=t, te=t_stop, dt=dt, tag_prefix='', tag_seperator='.',
                                       timeout=timeout)
            while True:
                if app.nc.terminate_event.is_set():
                    app.status = 0
                    break
                if app.nc.hibernate_event.is_set():
                    app.status = 2
                    break

                try:
                    data = input.get_at_time(t)
                except (DataSourceHibernating, TimeoutError) as e:
                    if app.allow_hibernation:
                        app.status = 2
                        app.nc.hibernate(message="hibernating")
                    else:
                        app.status = -1
                    details = tb_str(e)
                    break
                except DataSourceCompleted as e:
                    app.status = 1
                    details = tb_str(e)
                    app.message = 'No more input data'
                    break
                except (DataSourceEmptyDataset, DataSourceStreamClosed) as e:
                    app.status = -1
                    details = tb_str(e)
                    break

                completed = 0
                if t_stop > 0:
                    completed = (1 - (t_stop - t) / (t_stop - t0)) * 100

                i+=1
                if app.backup.is_set():
                    app.save_states(t, 'backup', f"scheduled checkpoint @ {datetime.now()}")
                    app.backup.clear()

                if data is None:
                    app.nc.set_scenario_progress(f"waiting for data. Last update: "
                                                  f"{datetime.fromtimestamp(t)}",
                                                  ScenarioStatus.RUNNING if not app.init else ScenarioStatus.WAITING,
                                                  completed)
                    if app.print_update(datetime.now().timestamp(), 0, 10):
                        self.logger.info(f"no data. Simulation time: {t}. ")
                        sleep(1)
                    continue

                self.system.update_inputs(data)

                if t_stop > 0:
                    app.nc.set_scenario_progress("running", ScenarioStatus.RUNNING, completed)

                if t==0 and data['_index'] > 0:
                    self.logger.warning(f'setting start time to {data["_index"]}')
                    t = data['_index']
                    t0 = t

                # if data is a dict, then convert to list

                if app.init:
                    app.nc.set_scenario_progress("building model", ScenarioStatus.MODEL_INITIALIZING, 0, force=True)
                    initial_output = self.system.initialize_model()
                    if initial_output:
                        if "_index" not in initial_output:
                            initial_output.update({'_index': 0})
                        app.output.write_row(initial_output)

                try:
                    tnew, outputs = self.step(t, dt)
                    if not outputs:
                        continue
                    if app.init:
                        app.nc.set_timeseries_meta_data([{"name": tag} for tag in outputs.keys()],
                                                         offset=app.start_time)
                        app.init = False
                    if not self.align_outputs_to_next_timestep:
                        outputs.update({'_index': t - app.start_time})
                    else:
                        outputs.update({'_index': tnew - app.start_time})

                except Exception as e:
                    details = tb_str(e)
                    raise SimulationError(details)

                # Advance time
                t = tnew
                # save output
                self.output.write_row(outputs)

                self.logger.info(f'Calculation step. Time is now {t}. completed: {completed}')
                last_data = data

                if (t >= t_stop) and (t_stop > 0):
                    self.logger.warning('maximum time reached')
                    app.status = 1
                    self.message = "Simulation completed"
                    break

        except Exception as e:
            self.logger.error(f'numerous_app crashed: {tb_str(e)}')
            self.logger.debug(f'previous data: {last_data}')
            self.logger.debug(f'data at crash: {data}')
            cause = 'exception'
            details = {"error message": tb_str(e), "previous_data": last_data, "data at crash": data}
            app.status = -1
            app.message = "app error - see logs"
        finally:
            if app.status == 0:
                app.terminate.set()
                cause = "forcefully terminated"
            elif app.status == 1:
                app.terminate.set()
                cause = "completed"
            elif app.status == 2:
                app.terminate.set()
                cause = 'hibernating'
            self.output.close()
            app.save_states(t, cause, details)
            self.logger.warning('job terminated')