import importlib

dynamic_imports = {}


def get_class(object_path):
    object_path_list = object_path.split('.')
    module_ = object_path_list[:-1]
    object_ = object_path_list[-1]

    file_ = module_[-1]

    location_ = "/".join(module_) + '.py'

    spec_ = importlib.util.spec_from_file_location(file_, location_)

    dynamic_imports[file_] = importlib.util.module_from_spec(spec_)
    spec_.loader.exec_module(dynamic_imports[file_])
    return getattr(dynamic_imports[file_], object_)


def get_item_from_list(list_, value, key='_id', return_none_not_found=False):
    if key == '_id':
        try:
            return list_[value]
        except KeyError:
            raise KeyError(f'Item with {key} == {value} not found!')

    for l in list_:
        if getattr(l, key) == value:
            return l
    else:
        if return_none_not_found:
            return None
        else:
            raise KeyError(f'Item with {key} == {value} not found!')


def get_key_from_all(dict_, key):
    for i in dict_.values:
        return getattr(i, key)


class Parameter:

    def __init__(self, _id):
        self._id = _id

class InputVariable:

    def __init__(self, _id):
        self._id = _id


class Component:

    def __init__(self, _id):
        self._id = _id
        self.parameters = {}
        self.input_variables = {}

    def add_parameter(self, parameter:Parameter):
        self.parameters[parameter._id] = parameter

    def add_input_variable(self, input_variable:InputVariable):
        self.input_variables[input_variable._id] = input_variable

    def get_parameter(self, value, key='_id'):
        return get_item_from_list(self.parameters, value, key)


def dummy_get_config(config):
    return {}

class Scenario:
    def __init__(self, config_source=dummy_get_config):
        self.components={}
        self.instanciated_components={}
        self._config = config_source

    def add_component(self, component:Component):
        self.components[component._id] = component

    def get_component(self, value, key='_id'):
        return get_item_from_list(self.components, value, key)

    def instanciate_components(self, repo_path, ignore_undefined_references=True):

        ordered = [None]*10000
        for c in self.components.values():

            ordered[c.order] = c
        self.components = {c._id: c for c in ordered if c is not None}


        for c_id, c in self.components.items():

            if c.isMainComponent and not c.disabled:
                kwargs={}
                #process parameters
                for p_id, p in c.parameters.items():
                    if p.type == 'reference':
                        if p.value in self.instanciated_components or not ignore_undefined_references:
                            kwargs[p_id] = self.instanciated_components[p.value]
                        else:

                            kwargs[p_id] = None
                    elif p.type == 'config':

                        conf = self._config(p.value)

                        for k, v in conf.items():
                            kwargs[k] = v

                    else:
                        kwargs[p_id] = p.value



                self.instanciated_components[c_id]=get_class(repo_path+'.'+c.item_class)(**kwargs)


def map_basic_attributes(obj_, dict_, allow_list=False, allow_dict=False):
    # Map parameters to scenario
    for k, v in dict_.items():
        if not k == 'id':
            if (allow_list or not isinstance(v, list)):
                if (allow_dict or not isinstance(v, dict)):
                    setattr(obj_, k, v)


def scenario_from_dict(scenario_dict, config_source=dummy_get_config):

    scenario = Scenario(config_source=config_source)

    map_basic_attributes(scenario, scenario_dict)

    #Make components list

    for c in scenario_dict['simComponents']:
        component = Component(_id=c['id'])

        map_basic_attributes(component, c)


        for p in c['parameters']:
            parameter = Parameter(_id=p['id'])

            map_basic_attributes(parameter, p, allow_list=True, allow_dict=True)

            component.add_parameter(parameter)

        for iv in c['inputVariables']:
            input_var = InputVariable(_id=p['id'])

            map_basic_attributes(input_var, iv, allow_list=True, allow_dict=True)

            component.add_input_variable(input_var)

        scenario.add_component(component)

    return scenario