import logging
import datetime
import os
import signal

from importlib.metadata import version
from time import sleep
from threading import Thread, Event

from .job import NumerousBaseJob
from .errors import tb_str
from numerous_cli.client import get_client
from numerous_api_client.client import NumerousClient
from numerous_api_client.client.numerous_client import ScenarioStatus

VERSION = version('numerous_image_tools')
APPVERSION = os.getenv('APPVERSION', "?")
APPNAME = os.getenv('APPNAME')
LOG_LEVEL = os.getenv('LOG_LEVEL', logging.INFO)
MODEL = os.getenv('MODEL')
MAX_RESTARTS = int(os.getenv('MAX_RESTARTS', 0))

local = os.getenv('KUBERNETES_SERVICE_HOST',"") == ""
logger = logging.getLogger('numerous-image-tools')
logger.setLevel(level=LOG_LEVEL)

class NumerousApp():
    def __init__(self, numerous_client: NumerousClient = None,
                 appname = "defaultnumerousApp",
                 max_restarts = 0, numerous_job: NumerousBaseJob = None, model_folder: str = None):
        self.appname = appname
        self.nc = numerous_client

        self.backup = Event()
        self.terminate = Event()
        self.output = None
        self.scenario = None
        self.model_definition = None
        self.files = None
        self.job_spec = None
        self.solver_settings = None
        self.sim_job = None
        self.subscribe = False
        self.solver_settings = None
        self.logger = None
        self.allow_hibernation = False
        self.start_time = None
        self.end_time = None
        self.max_restarts = max_restarts
        self.status = 0
        self.message = ""
        self.logger = logging.getLogger(self.appname)
        self.logger.setLevel(level=LOG_LEVEL)
        self.model_folder = model_folder
        self.numerous_job = numerous_job

        self.setup_scenario()

    def setup_scenario(self):

        self.scenario, self.model_definition, self.files = self.nc.get_scenario()
        self.nc.set_scenario_progress('initializing', ScenarioStatus.ENVIRONMENT_INITIALIZING, 0.0, force=True)

        # Prepare simulation setup

        self.start_time = self.nc.run_settings.start.timestamp()
        self.end_time = self.nc.run_settings.end.timestamp()
        self.appname = self.scenario['scenarioName']
        self.allow_hibernation = self.nc.params.get('allow_hibernation', 'False')
        self.logger.info(f'allow hibernation: {self.allow_hibernation}')

    def loop(self):

        self.logger.info(f'starting numerous app for {self.appname}')
        thread = Thread(target=self.numerous_job._start, args=(self,), daemon=False)
        thread.start()
        starttime = datetime.datetime.now()
        last_checkpoint = starttime
        restarts = 0
        warned = False

        while not self.terminate.is_set():
            checkpoint_time = datetime.datetime.now()
            if not thread.is_alive() and self.status == -1:
                if not warned:
                    self.nc.set_scenario_progress(message=
                                                  f"{self.message}. Retrying after 5 mins "
                                                  f"({restarts + 1}/{self.max_restarts})",
                                                  status=ScenarioStatus.RUNNING, force=True)
                    warned = True
                if (datetime.datetime.now() - starttime > datetime.timedelta(seconds=300)) and \
                        restarts < self.max_restarts:
                    self.logger.error(f'{self.appname} is dead.. restarting')
                    self.status = 0
                    self.message = ""
                    thread = Thread(target=self.numerous_job._start, args=(self, ), daemon=False)
                    thread.start()
                    sleep(5)
                    restarts +=1
                    starttime = datetime.datetime.now()
                    warned = False

                elif restarts >= self.max_restarts:
                    self.logger.error(f'{self.appname} is dead and reached max restarts')
                    break

            if checkpoint(checkpoint_time, last_checkpoint):
                self.backup.set()
                last_checkpoint = checkpoint_time

            sleep(0.5)
        self.logger.debug(f"waiting for {self.appname} to shut down...")
        thread.join()
        self.logger.info(f"{self.appname} stopped")
        self.logger.info("Main thread stopped gracefully")




def checkpoint(tnow, tlast, interval=60):
    dt=datetime.timedelta(minutes=interval)
    if tnow-tlast > dt:
        return True
    else:
        return False

def run_job(numerous_job=None, appname=None, max_restarts=0, model_folder="models"):
    """
    Runs the simulation job with the model located in model_folder_name. Handles all events.
    """

    def handler(sig, frame):
        logger.debug(f'main thread received {sig} {frame}')

    signal.signal(signal.SIGINT, handler)
    signal.signal(signal.SIGTERM, handler)

    nc = None
    try:
        RESET_JOB = os.getenv('RESET_JOB', 'False') == "True"

        if local:
            nc = get_client(reset_job=RESET_JOB)
        else:
            nc = NumerousClient()

        nc.logger.setLevel(LOG_LEVEL)

        if numerous_job is None:
            raise KeyError("no simulation job specified")
        if appname is None:
            logger.warning("no appname specified")

        logger.info(f"Welcome to numerous image tools. Base version {VERSION} - app {APPNAME} version {APPVERSION}")

        app = NumerousApp(numerous_client=nc, appname=appname,
                          max_restarts=max_restarts, numerous_job=numerous_job, model_folder=model_folder)

        app.loop()

        if app.status == -1:
            nc.set_scenario_progress(app.message, ScenarioStatus.FAILED, 0, force=True)
        elif app.status == 0:
            nc.set_scenario_progress("interrupted", ScenarioStatus.FINISHED, force=True)
        elif app.status == 1:
            nc.set_scenario_progress(app.message, ScenarioStatus.FINISHED, 100, force=True)
        elif app.status == 2:
            nc.set_scenario_progress("hibernating", ScenarioStatus.HIBERNATING, force=True)
        else:
            nc.set_scenario_progress("unknown", ScenarioStatus.FINISHED, force=True)

        logger.debug(f'setting state {app.status}')
        nc.state.set('_status', app.status)

    except Exception as e:
        logger.error(f"unhandled exception: {tb_str(e)}")
        if nc is not None:
            nc.set_scenario_progress("unhandled error", ScenarioStatus.FAILED, 0.0, force=True)
            logger.debug(f'setting state -2')
            nc.state.set('_status', -2)
    finally:
        if nc is not None:
            nc.close()








