import pytest
from numerous_api_client.client import NumerousClient, open_client
from datetime import datetime, timedelta
from uuid import uuid4
import json
import pandas as pd
import pytz
from time import time, sleep
import threading

tzl = pytz.utc

data_length = 10000
write_cols = 10

#Generate some data for this test
data = {'c'+str(i): [j for j in range(8760)] for i in range(write_cols)}

df = pd.DataFrame(data)
df.dropna(inplace=True, axis='columns')

# Reduce data
df = df[list(df)[:write_cols]]
df = df.head(data_length)
df['_index'] = [(datetime(2020, 1, 1, tzinfo=tzl) + timedelta(hours=i)).timestamp() for i in range(len(df))]

project = "1aXn9A711vkR6sjvFn8G"#str(uuid4())
scenario = "nOTJChsTHP5fhlM7UfaG"#str(uuid4())

@pytest.fixture(scope='session')
def client():

    return NumerousClient(project=project, scenario=scenario)

@pytest.fixture(scope='session')
def meta():
    return dict(start=datetime(2021,1,1), tags=['c' + str(i) for i in range(10)],  aliases={'a1': ['c1']})


def test_set_get_custom_meta_data(client):
    custom_meta = "test_meta"
    client.set_timeseries_custom_meta_data(custom_meta)

    read_back_meta = client.get_timeseries_custom_meta_data()

    assert custom_meta == read_back_meta, custom_meta +" != "+read_back_meta


def test_set_get_state(client):
    state = {'state': 1}
    client.set_state(state)

    read_back_state = client.get_state()
    print('state read: ', read_back_state)
    assert state == read_back_state,  state +" != "+read_back_state


def test_write_data(client: NumerousClient, meta):

    client.clear_timeseries_data()
    tic = time()
    client.data_write_df(df)
    toc =time()
    print('wrote df in: ',toc-tic)

    df_read = client.data_read_df(tags=[])

def test_get_meta_data(client, meta):
    read_back_meta = client.get_timeseries_meta_data()
    assert read_back_meta.tags is not None, "No tags!"
    assert read_back_meta.start is not None, "No start!"

def test_read_data_unbound_time_range(client: NumerousClient, meta):

    stream = client.read_data_stream(tags=['_index'])

    count = 0
    for r in stream:
        count += 1

    assert count > 0, 'No matching rows returned!'

def test_read_data_bound_time_range(client: NumerousClient, meta):
    start = datetime(2019, 1, 2, 0, tzinfo=tzl)
    end = datetime(2020, 1, 2, 10, tzinfo=tzl)
    stream = client.read_data_stream(tags=['_index'], start=start, end=end)

    count=0
    for r in stream:
        count+=1
        assert r[0]=='_index', 'Only asked for index col!'
        read_data = [datetime.fromtimestamp(t, tz=tzl) for t in r[1]]
        for d in read_data:
            assert min(read_data)>= start, 'data starting before start '+str(min(read_data))
            assert min(read_data) < end, 'data later than end '+str(min(read_data))

    assert count>0, 'No matching rows returned!'

def test_clear_data(client: NumerousClient):
    client.clear_timeseries_data()

    read_back_meta = client.get_timeseries_meta_data()

    assert  read_back_meta.scenario == "", "tags still present!"

    read_back_custom_meta = client.get_timeseries_custom_meta_data()

    assert read_back_custom_meta == "", "Custom meta not removed!"

    df_read = client.data_read_df(tags=[])
    assert df_read.equals(pd.DataFrame()), 'Data not removed.'


def test_subscribed_reader():
    rows_written = []
    tic = time()

    def producer(n):

        with open_client(project, scenario) as client:

            with client.open_writer(clear=True, buffer_size=1) as writer:
                print('writing started')
                for i in range(n):
                    print('write:', time())
                    sleep(.1)
                    r= {'_index': i * 3600, 'P': i}
                    writer.write_row(r)
                    rows_written.append(r)
            toc = time()
            print('Wrote rows in: ',toc-tic, 's')

    rows_read = []
    def listener():

        count = 0
        with open_client(project, scenario) as client:
            print('reader started')

            # Read data by streaming row by row
            for row in client.read_data_as_row(subscribe=True):
                print('read: ', time())
                rows_read.append(row)
                count += 1

            toc = time()
            print('Read rows in: ', toc - tic, 's')

            print('shutting down listener')


    print('clear')
    with open_client(project, scenario) as client:
        client.clear_timeseries_data()

        print('listening')
        listener_thread = threading.Thread(target=listener)
        listener_thread.start()

        print('Producing')

        producer_thread = threading.Thread(target=producer, args=(20,))
        producer_thread.start()

        print('producing started')

        producer_thread.join()
        listener_thread.join()
        print(rows_read)
        print(rows_written)
        assert rows_read == rows_written, 'Rows read != rows written'

        client.clear_timeseries_data()


def test_get_scenario(client: NumerousClient):
    client.get_scenario_document()

#def test_listen_scenario(client: NumerousClient):
#    for d in client.listen_scenario_document():
#        print(d)

def test_upload_scenario_file(client: NumerousClient):
    with open('./test.txt', 'w') as f:
        f.write('testingtesting')
    client.upload_file('./test.txt','test.txt')

def test_set_scenario_progress(client: NumerousClient):
    client.set_scenario_progress(message="client in progress", status="running", clean=True, progress=55.0)


def test_push_scenario_error(client: NumerousClient):
    client.push_scenario_error(error="client error test")


def test_push_scenario_log(client: NumerousClient):
    client.push_scenario_log(message="client log test")

def test_set_scenario_results(client: NumerousClient):
    results_in = (['a', 'b'], [1.0, 2.0], ['kW', 'm/s'])
    client.set_scenario_results(*results_in)
    results = client.get_scenario_results()
    assert results == results_in, 'Results in and out different'

def test_clear_scenario_results(client: NumerousClient):
    client.clear_scenario_results()


#def test_push_scenario_formatted_error(client: NumerousClient):
#    client.push_scenario_formatted_error("test formatted error", "this is a test", "test category",  exception_object_type=None, exception_object_message="?", full_traceback="traceback", initialize=False)
#def test_delete_scenario(client: NumerousClient):
#    client.delete_scenario()

def test_download_scenario_files(client: NumerousClient):
    files = client.get_download_files(files=['test.txt'])
    print('Downloaded files: ',files)
    n_files = len(list(files.keys()))
    assert n_files==1, f'Didnt download 1 file, but {n_files} files!'

"""
@pytest.mark.asyncio
async def test_writer(client: NumerousClient, meta):
    df = pd.read_csv('./dump.csv', sep=';', decimal='.')
    df.dropna(inplace=True, axis='columns')
    df.pop('Unnamed: 0')

    data_length = 10
    write_cols = 10

    # Reduce data
    df = df[list(df)[:write_cols]]
    df = df.head(data_length)

    df['_index'] = [datetime(2020, 1, 1,tzinfo=tzl) + timedelta(hours=i) for i in range(len(df))]
    df.set_index('_index', inplace=True)
    tic = time()
    async with client.open_writer(scenario, buffer_size=10000) as writer:
        count=0
        total=len(df)
        for r in df.iterrows():
            count+=1
            if count%1000==0:
                print('prog: ', count/total*100,'%')

            r[1]['_index']=r[0].timestamp()
            await writer.write_line(r[1])
    toc = time()
    print('write took: ',toc-tic)
    tic=time()
    #df_read =  await client.data_read_df(tags=[], start=datetime(2000, 1, 1, ))
    toc=time()
    print('read took: ', toc-tic)
    
"""