from numerous_api_client.client import open_client
import os
from job_spec_formatter import format_job_spec, item_key_value

from time import time
import logging
import datetime
import traceback
from dotenv import load_dotenv
import random
from time import sleep

load_dotenv()

log = logging.getLogger('main.simulation')
log.setLevel(logging.DEBUG)
def build_model_from_spec(job_id, scenario, model_definition, get_state, code_path):
    log.debug('Building model!')

    spec, run_settings = format_job_spec(job_id, scenario, model_definition)
    log.debug('spec ready')
    # Perform imports of code
    parameters = scenario['parameters']

    def get_val(param):
        return item_key_value(parameters, 'id', param)['value']

    # Read the stub (containing simulation data and model components) into a model objec

    old_state_applied = False
    state = get_state()
    if state is not None and 't' in state:
         log.info("Applying saved state!")
         old_state_applied = True
    else:
        log.info("No state applied.")

    return spec, state, old_state_applied, run_settings, spec['sim_specification']['sim_spec']['t_0']

class Debouncer:

    def __init__(self, debounce):
        self.debounce = debounce
        self.last = -999999

    def __call__(self, f, *args, **kwargs):
        tic = time()
        if tic > self.last+self.debounce:
            self.last = tic
            f(*args, **kwargs)

def simulate_model(scenario, model_definition, output, client, code_path, timeout):
    client.set_scenario_progress('Simulation initializing', 'initializing', 0.0, force=True)


    spec, state, state_loaded, run_settings, t0 = build_model_from_spec(client._job_id, scenario, model_definition, client.get_state, code_path)
    log.debug('Retrieved spec and model.')

    pause = 0


    for p in scenario['parameters']:
        if p['id'] == 'pause':
            pause =p['value']

    print('pause: ', pause)

    sim_spec = spec['sim_specification']['sim_spec']

    log.debug('simspace: ' + str(sim_spec))

    dt = sim_spec['dt']
    t_end = sim_spec['t_end']

    log.debug('t0: '+ str(t0))
    log.debug('t end : ' + str(t_end))

    if 't' in state:
        t=state['t']
    else:
        t=0

    dt_int = sim_spec['dt_internal']
    print('Starting date: '+str(run_settings['startDate']))
    offset = datetime.datetime.fromisoformat(run_settings['startDate']).timestamp()

    def apply_row(row):
        last_t = row['_index']
        row_ix_offset = row['_index_relative']
        row_date = datetime.datetime.fromtimestamp(last_t)

        row['t_of_year'] = (row_date -datetime.datetime(row_date.year,1,1)).total_seconds()


        pop_keys = ['_index', '_index_relative', '_datetime_utc', 't']

        for pk in pop_keys:
            if pk in row:
                row.pop(pk)

        return last_t

    first_call = {'is_first': True}

    def output_(data):
        data['_index']=data['t']# + 1609459200
        output.write_row(data)

        if first_call['is_first']:
            log.debug('First index written: '+str(data['_index']))
            log.debug('Offset: ' + str(offset))
            log.debug('Start date: '+str(datetime.datetime.fromtimestamp(offset)))

            first_call['is_first'] = False

            client.set_timeseries_meta_data([{'name': t} for t in data.keys()], {}, offset=offset, timezone='UTC', epoch_type='s')

    def set_state_(force=False):
        state = {'t': t}
        client.set_state({'system_state': state}, force=force)

    def progress():
        return round((t- t0) / (t_end - t0) * 100, 0)

    client.set_scenario_progress('Simulation in progress', 'running', progress(), force=True)

    debounce = Debouncer(300)
    debounce(set_state_, True)

    log.debug('Starting simulation')
    log.debug('Start time: ' + str(datetime.datetime.fromtimestamp(t0+offset)))
    log.debug('End time: ' +str(datetime.datetime.fromtimestamp(t_end+offset)))

    try:
        rows = 0

        log.debug('Starting at t: ' + str(datetime.datetime.fromtimestamp(t0 + offset)))

        inputs = client.get_inputs(scenario, t0=t0+offset, te=t_end+offset+dt, tag_prefix='Stub', tag_separator='.', timeout=timeout)
        while True:
            rows+=1
            row = inputs.get_at_time(t+offset)

            last_t = apply_row(row)
            t += dt
            row.update({'t': t, 'out1': random.random()})

            output_(row)
            client.set_scenario_progress('Test in progress', 'running', progress())

            debounce(set_state_)


            if t >= t_end:
                log.debug(f't end of {t_end} s reached')

                break
            sleep(float(pause))


        log.debug(f'Test finished at t: {t} s')
        client.set_scenario_progress('Test completed', 'finished', 100.0, force=True)
        #client.upload_file('./Alingsas_Alingsas_102204_2019_Sveby.csv', 'Alingsas_Alingsas_102204_2019_Sveby.csv')

    except TimeoutError:
        client.set_scenario_progress('Test hibernated', 'finished', progress(), force=True)

    except (KeyboardInterrupt):
        client.set_scenario_progress('Test terminated', 'terminated', progress(), force=True)

    except Exception as e:
        print(rows)
        print(t/3600)
        print(t_end/3600)
        tb = traceback.format_exc()
        print(tb)
        log.error(tb)
        client.set_scenario_progress('Test failed', 'failed', progress(), force=True)


    finally:
        set_state_(force=True)

def run(job_id=None, project=None, scenario=None, workpath="."):

    #Open the numerous client
    with open_client(job_id=job_id, project=project, scenario=scenario) as client:
        try:

            scenario, model_definition, files = client.get_scenario(path=workpath)
            os.chdir(workpath)

            with client.open_writer() as output:
                log.debug('Writer opened!')

                simulate_model(scenario=scenario, model_definition=model_definition, output=output, client=client, code_path="", timeout=120)
                log.debug('Test completed')

        except (KeyboardInterrupt):
            client.set_scenario_progress('Test terminated', 'terminated', force=True)

        except Exception:
            tb = traceback.format_exc()
            log.error(tb)
            client.set_scenario_progress('Test failed', 'failed', force=True)

if __name__ == '__main__':
    run()