import time
from typing import Any, List

import pytest
from pytest_mock import MockerFixture

from numerous.client import DataSourcesReader, NumerousClient
from numerous.client.data_source import DataSourceCompleted, DataSourceEmptyDataset, DataSourceHibernating, \
    DynamicSource, InputManager


@pytest.fixture
def input_manager(mocker: MockerFixture):
    return InputManager([mocker.MagicMock(DynamicSource)], t=0, t0=0)


@pytest.fixture
def client(mocker: MockerFixture, input_manager):
    client = mocker.MagicMock(NumerousClient)
    client.get_inputs.return_value = input_manager
    client.hibernate_event = mocker.MagicMock()
    client.terminate_event = mocker.MagicMock()
    return client


def test_return_none_on_completed(mocker: MockerFixture, input_manager, client):
    client.hibernate_event.is_set.return_value = False
    client.terminate_event.is_set.return_value = False
    reader = DataSourcesReader(client=client, scenario_data={}, start_time=0, t0=1, te=3, dt=1,
                               tag_prefix='', tag_separator='.', timeout=10)

    data_values: List[Any] = [{'param': 2}, {'param': 4}, {'param': 6}]
    calls_after = 2
    input_manager.get_at_time = mocker.MagicMock(side_effect=data_values + [DataSourceCompleted] * calls_after)
    actual = [reader.read(t) for t in range(len(data_values) + calls_after)]
    expected_values = data_values + [None] * calls_after

    assert actual == expected_values


def test_hibernates_if_start_time_equals_to_t0(mocker: MockerFixture, input_manager, client):
    client.hibernate_event.is_set.return_value = False
    client.terminate_event.is_set.return_value = False
    reader = DataSourcesReader(client=client, scenario_data={}, start_time=0, t0=0, te=3, dt=1,
                               tag_prefix='', tag_separator='.', timeout=10)

    input_manager.get_at_time = mocker.MagicMock(side_effect=DataSourceHibernating)
    data = reader.read(0)

    assert data is None
    assert client.hibernate.called_once()


def test_sleeps_and_reinits_reader_if_start_time_less_t0(mocker: MockerFixture, input_manager, client):
    client.hibernate_event.is_set.return_value = False
    client.terminate_event.is_set.return_value = False
    reader = DataSourcesReader(client=client, scenario_data={}, start_time=0, t0=1, te=3, dt=1,
                               tag_prefix='', tag_separator='.', timeout=10)

    retry_delay = 0.2
    reader.set_retry_delay(retry_delay)
    calls_count = 20
    data_value = {'param': 1.0}
    side_effect: List[Any] = [DataSourceHibernating] * calls_count
    side_effect.append(data_value)
    input_manager.get_at_time = mocker.MagicMock(side_effect=side_effect)

    start_time = time.time()
    data = reader.read(1)
    execution_time = time.time() - start_time

    assert retry_delay * calls_count < execution_time
    assert data == data_value
    assert client.get_inputs.call_count == 1 + calls_count  # one on init + one per retry attempt


def test_sleeps_and_reinits_reader_if_empty(mocker: MockerFixture, input_manager, client):
    client.hibernate_event.is_set.return_value = False
    client.terminate_event.is_set.return_value = False
    reader = DataSourcesReader(client=client, scenario_data={}, start_time=0, t0=1, te=3, dt=1,
                               tag_prefix='', tag_separator='.', timeout=10)

    retry_delay = 0.2
    retry_timeout = 3
    calls_count = int(retry_timeout / (2 * retry_delay))
    reader.set_retry_delay(retry_delay)
    reader.set_retry_timeout(retry_timeout)
    data_value = {'param': 1.0}
    side_effect: List[Any] = [DataSourceEmptyDataset] * calls_count
    side_effect.append(data_value)
    input_manager.get_at_time = mocker.MagicMock(side_effect=side_effect)

    start_time = time.time()
    data = reader.read(1)
    execution_time = time.time() - start_time

    assert retry_delay * calls_count < execution_time
    assert data == data_value
    assert client.get_inputs.call_count == 1 + calls_count  # one on init + one per retry attempt


def test_none_on_empty_timeout(mocker: MockerFixture, input_manager, client):
    client.hibernate_event.is_set.return_value = False
    client.terminate_event.is_set.return_value = False
    reader = DataSourcesReader(client=client, scenario_data={}, start_time=0, t0=1, te=3, dt=1,
                               tag_prefix='', tag_separator='.', timeout=10)

    retry_delay = 0.2
    retry_timeout = 3
    calls_count = int(2 * retry_timeout / retry_delay)
    reader.set_retry_delay(retry_delay)
    reader.set_retry_timeout(retry_timeout)
    data_value = {'param': 1.0}
    side_effect: List[Any] = [DataSourceEmptyDataset] * calls_count
    side_effect.append(data_value)
    input_manager.get_at_time = mocker.MagicMock(side_effect=side_effect)

    start_time = time.time()
    data = reader.read(1)
    execution_time = time.time() - start_time

    assert retry_timeout < execution_time
    assert data is None
    assert (client.get_inputs.call_count-1) <= calls_count / 2 < calls_count  # one on init + one per retry attempt


def test_interrupts_on_client_hibernate_or_terminate(mocker: MockerFixture, input_manager, client):
    reader = DataSourcesReader(client=client, scenario_data={}, start_time=0, t0=1, te=3, dt=1,
                               tag_prefix='', tag_separator='.', timeout=10)

    client.hibernate_event.is_set.return_value = True
    client.terminate_event.is_set.return_value = False
    input_manager.get_at_time = mocker.MagicMock()

    data = reader.read(1)
    assert data is None
    input_manager.get_at_time.assert_not_called()

    client.hibernate_event.is_set.return_value = False
    client.terminate_event.is_set.return_value = True

    data = reader.read(1)
    assert data is None
    input_manager.get_at_time.assert_not_called()
