import os
from contextlib import contextmanager
from unittest.mock import MagicMock

from pytest_mock import MockerFixture

import spm_pb2

from numerous_api_client.client import NumerousClient


def test_upload_file(mocker: MockerFixture, mock_spm: MagicMock):
    mock_post = mocker.patch('requests.post')
    mock_spm().GetScenarioCustomMetaData.return_value = spm_pb2.ScenarioCustomMetaData(meta='{}')
    mock_spm().GenerateScenarioUploadSignedURL.return_value = spm_pb2.ScenarioUploadURL(scenario='scenario_id',
                                                                                        url='http://upload.url')
    with open('test_file.txt', 'wb') as file:
        file.write(b'Test file content')

    client = NumerousClient('job_id', 'project_id', 'scenario_id', 'https://server', port=50000, secure=True,
                            refresh_token='refresh')
    try:
        client.upload_file('test_file.txt', 'test_file')
    finally:
        os.remove('test_file.txt')
        client.close()

    assert mock_post.call_args.args[0] == 'http://upload.url'
    assert mock_post.call_args.kwargs['data'].read() == b'Test file content'
    mock_spm().GenerateScenarioUploadSignedURL.assert_called_once()


def test_upload_file_path(mocker: MockerFixture, mock_spm: MagicMock, mock_file_manager: MagicMock):
    uploaded_content = None
    mock_post = mocker.patch('requests.post')
    mock_spm().GetScenarioCustomMetaData.return_value = spm_pb2.ScenarioCustomMetaData(meta='{}')
    mock_file_manager().GetUploadUrls.return_value = spm_pb2.FileUrls(urls=[spm_pb2.FileUrl(path='path/test_file.txt',
                                                                                            url='http://upload.url')])

    def post_side_effect(*args, **kwargs):
        nonlocal uploaded_content
        uploaded_content = kwargs['data'].read()
    mock_post.side_effect = post_side_effect

    with open('test_file.txt', 'wb') as file:
        file.write(b'Test file content')

    client = NumerousClient('job_id', 'project_id', 'scenario_id', 'https://server', port=50000, secure=True,
                            refresh_token='refresh')
    try:
        client.upload_file_path('path/test_file.txt', 'test_file.txt')
    finally:
        os.remove('test_file.txt')
        client.close()

    mock_file_manager().GetUploadUrls.assert_called_once()
    assert mock_post.call_args.args[0] == 'http://upload.url'
    assert uploaded_content == b'Test file content'


def test_download_file_path(mocker: MockerFixture, mock_spm: MagicMock, mock_file_manager: MagicMock):
    mock_get = mocker.patch('requests.get')

    @contextmanager
    def mock_response():
        mock = mocker.Mock()
        mock.iter_content.return_value = [b'Mocked file content']
        yield mock

    mock_get.return_value = mock_response()
    mock_spm().GetScenarioCustomMetaData.return_value = spm_pb2.ScenarioCustomMetaData(meta='{}')
    mock_file_manager().GetDownloadUrls.return_value = spm_pb2.FileUrls(urls=[
        spm_pb2.FileUrl(path='path/test_file.txt', url='http://download.url')
    ])

    client = NumerousClient('job_id', 'project_id', 'scenario_id', 'https://server', port=50000, secure=True,
                            refresh_token='refresh')
    try:
        client.download_file_path('path/test_file.txt', 'downloaded_file.txt')

        with open('downloaded_file.txt', 'rb') as file:
            downloaded_content = file.read()
    finally:
        if os.path.exists('downloaded_file.txt'):
            os.remove('downloaded_file.txt')
        client.close()

    assert downloaded_content == b'Mocked file content'
    assert mock_get.call_args.args[0] == 'http://download.url'
