import io
from logging import Logger
from pathlib import Path

import grpc
import requests
from spm_pb2 import HistoryRequest, PullRequest

from ..auth import AuthenticationError, TokenAuth, login
from ..repository import NumerousRepository, ProjectScenario, RepositoryRemote
from ..utils import bold, cyan, extract_gzipped_tarball, get_build_manager, green, red


def cleanup(path: Path):
    (path / NumerousRepository.FILE_NAME).unlink(True)
    (path / '.exclude').unlink(True)
    path.rmdir()


def command_clone(log: Logger, scenario: ProjectScenario, path: Path, remote: RepositoryRemote):
    path = path or Path.cwd()

    if path.exists():
        print(red(f'Cannot clone: {bold(path)} already exists'))
        return

    repo = NumerousRepository(path)
    repo.remote = remote
    repo.scenario = scenario
    path.mkdir()
    repo.save()

    try:
        access_token = login(repo, scenario)
    except AuthenticationError:
        print(red("Cannot clone: Login failed."))
        cleanup(path)
        return
    except Exception:
        print(red("Cannot clone: Error occured."))
        cleanup(path)
        log.debug("An error occured during clone.", exc_info=True)
        return

    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))
    with get_build_manager(repo.remote.api_url) as build_manager:
        history_request = HistoryRequest(repository=remote.name, scenario_id=scenario.id)
        history_reply, _call = build_manager.GetHistory.with_call(history_request, credentials=call_credentials)

        if not history_reply.updates:
            print(red('Cannot clone: No commit found'))
            return

        commit = history_reply.updates[0]
        repo.snapshot = commit.snapshot_id
        repo.scenario = ProjectScenario(scenario_id=commit.scenario_id, project_id=commit.project_id)

        print(cyan(f'Downloading snapshot {bold(repo.snapshot)} into {bold(path)}'))
        pull_request = PullRequest(repository=repo.remote.name, snapshot_id=commit.snapshot_id)
        snapshot_reply, _call = build_manager.PullSnapshot.with_call(pull_request, credentials=call_credentials)
        response = requests.get(snapshot_reply.archive_url.url, allow_redirects=True)
        extract_gzipped_tarball(io.BytesIO(response.content), path)

        repo.commit = commit.id
        repo.save()
        print(green(f"Repository {bold(repo.remote.name)} was cloned into {bold(path)}"))
