import logging
from numerous_api_client.client import open_client
from time import time, sleep
import math


logging.basicConfig(level=logging.NOTSET)

log = logging.getLogger('enumerous.main')
log.setLevel(logging.NOTSET)


class Debouncer:

    def __init__(self, debounce):
        self.debounce = debounce
        self.last = -999999

    def __call__(self, f, *args, **kwargs):
        tic = time()
        if tic > self.last+self.debounce:
            self.last = tic
            f(*args, **kwargs)


def run(project=None, scenario=None, code_path=None, clear_data=None):
    log.info('Starting run method')

    # Open the numerous client
    with open_client(project=project, scenario=scenario, clear_data=clear_data) as client:
        log.info('Client opened')

        try:
            scenario, model_definition, files = client.get_scenario()
            log.info('Scenario data retrieved')

            # Create a new output writer
            with client.open_writer() as output:
                log.info('Set scenario progress to initializing')
                client.set_scenario_progress('Simulation initializing', 'initializing', 0.0, force=True)

                def apply_row(row):
                    last_t = row['_index']
                    row.pop('_index')

                    return last_t

                first_call = {'is_first': True}

                def output_(data):
                    data['_index'] = data['t']
                    output.write_row(data)

                    if first_call['is_first']:
                        first_call['is_first'] = False
                        client.set_scenario_data_tags(data.keys())

                def set_state_(force=False):
                    state = {'nothing': 'important'}
                    client.set_state({'system_state': state}, force=force)

                t = 0
                t_end = 1000
                timeout = 60
                dt = 1

                def progress():
                    return round(t / t_end * 100, 0)

                client.set_scenario_progress('Simulation in progress', 'running', progress(), force=True)

                debounce = Debouncer(300)
                debounce(set_state_, True)

                try:
                    last_t = None
                    for row in client.iter_inputs(scenario, t0=t, dt=dt, tag_prefix='Stub', tag_seperator='.',
                                                  timeout=timeout):

                        if last_t is not None:
                            dt_solve = row['_index'] - last_t

                            t += dt_solve
                            output_({'t': t, 'var1': (math.sin(t/10)*(t/10))+abs(t/10)})
                            sleep(.1)
                            client.set_scenario_progress('Simulation in progress', 'running', progress())

                            debounce(set_state_)

                            if t >= t_end:
                                break

                        last_t = apply_row(row)

                    client.set_scenario_progress('Simulation completed', 'finished', 100.0, force=True)
                except TimeoutError:
                    log.info('Keyboard interrupt. Hibernating.')
                    client.set_scenario_progress('Simulation hibernated', 'finished', progress(), force=True)

                finally:
                    set_state_(force=True)

        except KeyboardInterrupt:
            log.info('Keyboard interrupt. Terminating.')

            client.set_scenario_progress('Simulation terminated', 'terminated', 0, force=True)
            raise

        except Exception:
            log.info('Exception caught. Scenario failed.')

            client.set_scenario_progress('Simulation failed', 'failed', 0, force=True)
            raise


if __name__ == '__main__':
    run()
