import json
import logging
from typing import Any, Dict, List, Optional, TypedDict

import spm_pb2
import spm_pb2_grpc

log = logging.getLogger(__name__)


class ScenarioSetting(TypedDict):
    path: List[str]
    type: str
    value: float
    id: str
    display_name: str


class OptimizationSetting:

    def __init__(self, path: List[str], setting_type: str, value: Any, raw: Any):
        self.path = path
        self.type = setting_type
        self.value = value
        self.raw = raw

    def scenario_setting(self, value: float) -> ScenarioSetting:
        return {'id': self.raw.get('id'), 'display_name': self.raw.get('displayName'), 'type': self.type,
                'value': value, 'path': self.path}


def extract_component(components: Dict[str, Any], component: Dict[str, Any]):
    value: Dict[str, Dict] = {'params': {}, 'components': {}}
    for param in component.get('parameters', []):
        value['params'][param['id']] = param.get('value')
    for subcomponent_ref in component.get('subcomponents', []):
        subcomponent = component.get(subcomponent_ref['uuid'], {})
        value['components'][subcomponent['name']] = extract_component(components, subcomponent)
    return value


class OptimizationConfiguration:

    def __init__(self, spm_stub: spm_pb2_grpc.SPMStub, project_id: str, scenario_id: str):
        scenario_response = spm_stub.GetScenario(spm_pb2.Scenario(project=project_id, scenario=scenario_id))
        scenario = json.loads(scenario_response.scenario_document)

        if scenario.get('optimize', False):
            raise RuntimeError(f"Optimization was not enabled on scenario {scenario_id} in project {project_id}.")

        # Load target scenario data
        self.target_scenario_id = scenario['optimizationTargetScenarioID']
        target_scenario_request = spm_pb2.Scenario(project=project_id, scenario=self.target_scenario_id)
        target_scenario_response = spm_stub.GetScenario(target_scenario_request)
        target_scenario = json.loads(target_scenario_response.scenario_document)
        self.target_job_id = next(job_id for job_id, job in target_scenario['jobs'].items() if job.get('isMain', False))

        # Validate and store optimizer configurations for parameters & inputs
        self.settings: List[OptimizationSetting] = []
        setting_paths = target_scenario.get('optimizationSettingsPaths', {})
        setting_paths.pop('__CONVERTED_FROM_MAP', None)  # Remove helper attribute
        settings_components = target_scenario.get('optimizationSettingsSimComponents', {})
        settings_components.pop('__CONVERTED_FROM_MAP', None)  # Remove helper attribute
        for settings_component_uuid, settings_components in settings_components.items():
            [settings_component] = settings_components  # Assumption that exactly one component exists
            setting_path = setting_paths.get(settings_component_uuid)
            setting_type = settings_component_uuid.split('_')[-1]  # param, value, offset, scaling
            setting_value = extract_component(settings_components, settings_component)
            component: Optional[Dict] = next((c for c in target_scenario.get('simComponents', [])
                                             if c['uuid'] == setting_path[-2]), None)
            if component is None:
                log.warning('Invalid component for optimization: %s', setting_path[-2])
                continue
            component_key = 'parameters' if setting_type == 'param' else 'inputVariables'
            setting_raw = next((attr for attr in component[component_key] if attr['uuid'] == setting_path[-1]), None)
            if None in (setting_path, setting_type, setting_value):
                log.warning('Invalid optimization setting component: %s', settings_component)
                continue
            self.settings.append(OptimizationSetting(setting_path, setting_type, setting_value, setting_raw))

        # Create a dictionary of components for extracting aggregator and goal function components
        components = {comp['uuid']: comp for comp in scenario.get('simComponents', [])}

        # Configure aggregator
        self.aggregator_config = None
        aggregator_component_uuid = next((comp['uuid'] for comp in components.values() if comp['name'] == 'aggregator'),
                                         None)
        if aggregator_component_uuid not in components:
            raise RuntimeError('Missing required aggregator component')

        aggregator_component = components.get(aggregator_component_uuid, {})
        self.aggregator_config = extract_component(components, aggregator_component)

        # Configure goal function
        self.goal_function_inputs = []
        self.goal_function_config = None
        goal_function_component_uuid = next((comp['uuid'] for comp in components.values()
                                            if comp['name'] == 'goal_function'), None)
        if goal_function_component_uuid not in components:
            raise RuntimeError('Missing required goal function component')

        goal_function_component = components.get(goal_function_component_uuid, {})
        self.goal_function_config = extract_component(components, goal_function_component)
        for input_variable in goal_function_component.get('inputVariables', []):
            if input_variable['dataSourceType'] != 'scenario':
                log.warning('Invalid input variable for optimization: %s', input_variable)
                continue
            self.goal_function_inputs.append({
                'tag': input_variable['tagSource']['tag'],
                'scale': input_variable['scaling'],
                'offset': input_variable['offset']
            })
