import datetime
import json
import logging

import grpc
from services import spm
from numerous_api_client.python_protos import spm_pb2_grpc, spm_pb2
from services.utilities import standard_error_handling, ErrorHandler, json_serial
import services.firebase as fire
from services.tokens import validated_request
from services.token_validation.validation import AccessLevel


log = logging.getLogger("numerous_api.server.build_manager")


class BuildManagerServicer(spm_pb2_grpc.BuildManagerServicer, ErrorHandler):

    def store_snapshot_metadata(self, request):
        if request.only_get_uploads:
            return
        file_struct = json.loads(request.file_structure)
        try:
            fire.create_snapshot('default', request.repository, request.snapshot_id, file_struct)
        except KeyError as e:
            log.error(e)
            self.set_error(KeyError,grpc.StatusCode.ALREADY_EXISTS, f'Snapshot {request.snapshot_id} already exist. Nothing to snapshot')

    def generate_snapshot_upload_url(self, request):
        path = f'file_registries/default/{request.repository}/{request.snapshot_id}'
        if not fire.check_file_exist(path):
            return fire.generate_resumable_upload_with_client(path, content_type="application/octet-stream")

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def CreateSnapshot(self, request, context):
        self.store_snapshot_metadata(request)
        upload_url = self.generate_snapshot_upload_url(request)
        return spm_pb2.SnapshotReply(upload_url=upload_url, snapshot_id=request.snapshot_id)

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def CreateHistoryUpdate(self, request, context):
        user_id = context.claims['user_id']
        commit_id = fire.create_history_update('default', request, user_id)
        return spm_pb2.HistoryReply(commit_id=commit_id)

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def GetHistory(self, request, context):
        log.debug('GetHistory: ' + str(request))
        user = fire.get_user(context.claims['user_id'])
        history = fire.get_history('default', request.repository, request.scenario_id, list(request.history_update_ids))
        return spm_pb2.History(updates=[spm_pb2.HistoryUpdate(
            repository = request.repository,
            scenario_id=h['scenario'],
            project_id=h['project'],
            snapshot_id=h['snapshot'],
            parent_commit_id=h['parent_commit_id'],
            comment=h['comment'],
            timestamp=h['timestamp'].timestamp() if isinstance(h['timestamp'], datetime.datetime) else h['timestamp'],
            user=spm_pb2.HistoryUser(
                id=context.claims['user_id'],
                email=user["mail"],
                full_name=user["fullName"]
            ),
            id=h['id']
        ) for h in history])


    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def PullSnapshot(self, request, context):
        signed_url = fire.generate_url(f'file_registries/default/{request.repository}/{request.snapshot_id}')
        file_signed_url = spm_pb2.FileSignedUrl(url=signed_url, name=f"{request.snapshot_id}.tar.gz")
        return spm_pb2.PullResponse(archive_url=file_signed_url, snapshot_id=request.snapshot_id)

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def LaunchBuild(self, request, context):
        build_id = spm.enqueue_build(request.repository, request.commit_id, request.project_id, request.scenario_id, request.job_id, request.launch_after_build)
        return spm_pb2.BuildReply(build_id=build_id)

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def DequeueBuild(self, request, context):
        json_ = "{}"

        return spm_pb2.Json(json=json_)

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def Builder(self, request, context):
        for r in spm.dequeue_build_request(request):
            yield spm_pb2.Json(json=r)

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def GetBuildInfo(self, request, context):
        return spm_pb2.BuildInfo(info=json.dumps(fire.get_build('default', request.repository, request.build_id), default=json_serial))

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def UpdateBuild(self, request, context):

        fire.update_build('default', request.repository, request.build_id, json.loads(request.update))
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def UpdateJobImageReference(self, request, context):
        fire.set_scenario_job_image(project_id=request.project, scenario_id=request.scenario, job_id=request.job, image_name=request.name, image_path=request.path, repository=request.repository, commit_id=request.commit_id, internal=request.internal, build=request.build)
        return spm_pb2.Empty()