from dotenv import load_dotenv
load_dotenv()

import os
import sys
import logging
import threading
import requests.exceptions

logging.basicConfig(level=logging.DEBUG)


from concurrent import futures
import time

import grpc
from services import spm, deployment
from numerous_api_client.python_protos import spm_pb2_grpc, spm_pb2, health_pb2_grpc
from services.tokens import validated_request
from services.token_validation.validation import AccessLevel, ValidationObjectType
from services import tokens as token_manager
import services.firebase as fire
import datetime
import json
from grpc_interceptor import ServerInterceptor
from grpc_reflection.v1alpha import reflection
from health_check_server import serve_health_check_endpoint

from services.deployment import Deploy
from services.utilities import ErrorHandler, standard_error_handling, json_serial

from build_manager_servicer import BuildManagerServicer
from health_servicer import HealthServicer

from firebase import Firebase
firebase_web_app_config = json.loads(os.environ["FIREBASE_WEB_APP_CONFIG"])
firebase = Firebase({
  "apiKey": firebase_web_app_config["api_key"],
  "authDomain": firebase_web_app_config["auth_domain"],
  "databaseURL": firebase_web_app_config["database_url"],
  "storageBucket": firebase_web_app_config["storage_bucket"]
})

api_server_cert = os.getenv('NUMEROUS_CERT_CRT')
my_own_url = str(os.getenv('NUMEROUS_API_SERVER'))
my_own_external_port = str(os.getenv('NUMEROUS_EXTERNAL_API_PORT'))
my_own_port = str(os.getenv('NUMEROUS_API_PORT'))
my_own_insecure_port = str(os.getenv('NUMEROUS_INSECURE_API_PORT'))
my_own_secure = str(os.getenv('SECURE_CHANNEL')) == "True"
build_image_url = ""


org = os.getenv('NUMEROUS_ORGANIZATION')

if org is None:
    raise ValueError('No organtization specified!')


debug = True


class InstanceInterceptor(ServerInterceptor):
    def intercept(self, method, request_or_iterator, context: grpc.ServicerContext, endpoint,  *args):

        context.metadata = {t[0]:t[1] for t in context.invocation_metadata()}
        log.debug('Requested endpoint: '+str(endpoint))
        return method(request_or_iterator, context)

    def intercept_service(self, continuation, handler_call_details):
        lines = [f"Intercept {handler_call_details.method} with metadata:"]
        lines += [f' * {m.key}={m.value}' for m in handler_call_details.invocation_metadata]
        log.debug("\n".join(lines))
        return continuation(handler_call_details)


def get_env(val, env, default=None):
    if val is None:
        env_val = os.getenv(env)
        if env_val is None:
            if default is None:
                raise KeyError(f'ENV var <{env}> is not set.')
            else:
                return default
        return env_val
    else:
        return val

log = logging.getLogger('numerous_api.server')
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
streamhandler = logging.StreamHandler(sys.stdout)
streamhandler.setFormatter(formatter)
log.addHandler(streamhandler)




def _get_job_channel(project_id, scenario_id, job_id):
    return ".".join(['COMMAND', project_id, scenario_id, job_id])



class SPMServicer(spm_pb2_grpc.SPMServicer, ErrorHandler):

    @validated_request(access_level=AccessLevel.ANY, validation_object_type=ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def Noop(self, request, context):

        return spm_pb2.NoopContent(scenario=request.scenario, content=request.content)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type=ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def ClearData(self, request, context):

        spm.clear_data(request.scenario, request.execution)
        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.WRITE, use_user_token=True, validation_object_type=ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def ClearDataTags(self, request, context):

        spm.delete_scenario_data(request.scenario, request.tags)
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.READ, validation_object_type=ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def ReadData(self, request: spm_pb2.ReadScenario, context):
        if request.execution:
            execution = request.execution

        else:
            execution = fire.get_latest_main_execution(request.project, request.scenario)
            if not execution:
                return spm_pb2.Empty()

        data_generator = spm.read_data(
            scenario=request.scenario, execution=execution, tags=list(request.tags),
            start=request.start, end=request.end, time_range=request.time_range, listen=request.listen
        )

        for d in data_generator:
            data_blocks = []
            blocks_size = 0
            for i, _d in enumerate(d[0]):
                new_block = spm_pb2.DataBlock(**_d)
                new_block_size = sys.getsizeof(new_block)
                if blocks_size + new_block_size > 4e6/70:
                    row_complete = d[1] if i +1 == len(d[0]) else False
                    block_complete = d[2] if i +1 == len(d[0]) else False

                    yield spm_pb2.DataList(scenario=request.scenario, execution=execution, data=data_blocks, row_complete=row_complete, block_complete=block_complete)

                    data_blocks=[]
                    blocks_size = 0

                blocks_size += new_block_size
                data_blocks.append(new_block)

            if len(data_blocks) > 0:
                yield spm_pb2.DataList(
                    scenario=request.scenario, execution=execution, data=data_blocks,
                    row_complete=d[1], block_complete=d[2]
                )
        log.debug(f'Reading data completed')

    """
    @validated_request(access_level=AccessLevel.WRITE)
    @standard_error_handling()
    def WriteData(self, request_iterator, context):
        wb = None

        for r in request_iterator:
            if wb is None:
                wb = spm.WriteBuffer(r.scenario, r.execution)
            wb.add_data(r.tag, r.values, r.row_complete, r.block_complete)

        if not wb is None:
            wb.close()

        return spm_pb2.Scenario(scenario=wb.scenario if wb is not None else "", execution=r.execution if wb is not None else "")
    """

    def write_data(self, request_iterator):
        wb = None
        validated = False
        first_req = True
        for r in request_iterator:

            if r.reset_block_counter:
                blockcounter = 0
            else:
                blockcounter = None

            if wb is None:
                wb = spm.WriteBuffer(r.scenario, r.execution, clear=r.clear, blockcounter=blockcounter)

            first_req = False

            if not validated:
                validated_request(access_level=AccessLevel.WRITE)(r)
                validated = True

            log.debug(f'Writing data for {r.scenario}, exe {r.execution}')

            len_ = -1
            if len(r.data)>0:
                for c in r.data:

                    if len_ < 0:
                        len_ = len(c.values)
                    else:
                        # assert (l_:=len(c.values)) == len_, f'All columns must have same lengths! {c.tag} {l_} != {len_}'
                        if not (l_ := len(c.values)) == len_:
                            self.set_error(ValueError, grpc.StatusCode.INVALID_ARGUMENT,
                                           f'All columns must have same lengths! {c.tag} {l_} != {len_}')



                _index_ok = r.ignore_index
                log.debug('Ignore ix: '+str(r.ignore_index))
                for c in r.data:

                    wb.add_data(c.tag, c.values)
                    if c.tag == '_index':
                        _index_ok = True

                log.debug('Ignoring ix: ' + str(_index_ok))
                if not _index_ok:
                    log.debug('Missing ix - adding it.')
                    wb.add_data('_index', [i for i in range(len_)])

            wb.complete(r.block_complete, r.row_complete, r.ignore_index)
            log.debug(f'Completed write. row_complete {r.row_complete}, block completed {r.block_complete}')

        if wb is not None:
            wb.close()

        return spm_pb2.Empty()

    @standard_error_handling()
    def WriteDataList(self, request_iterator, context):
        return self.write_data(request_iterator)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type=ValidationObjectType.SCENARIO, use_user_token=True)
    @standard_error_handling()
    def PushDataList(self, request, context):
        log.debug('Pushing datalist - ignoring index? '+str(request.ignore_index))
        return self.write_data([request])

    @validated_request(access_level=AccessLevel.READ, validation_object_type=ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetScenario(self, request, context):
        scenario_document_string, files = spm.read_scenario(request.project, request.scenario)

        return spm_pb2.ScenarioDocument(scenario=request.scenario, scenario_document=scenario_document_string, files=files)

    @validated_request(access_level=AccessLevel.READ, validation_object_type=ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetScenarioResultDocument(self, request, context):
        if request.execution is None or request.execution=="":
            request.execution = fire.get_latest_main_execution(request.project, request.scenario)
        result = fire.get_result_document(request.project, request.scenario, request.execution)
        return spm_pb2.ScenarioResultsDocument(project=request.project, scenario=request.scenario, execution=request.execution, result=json.dumps(result))

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type=ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def SetScenarioResultDocument(self, request, context):
        fire.set_result_document(request.project, request.scenario, request.execution, json.loads(request.result))
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.READ, validation_object_type=ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetGroup(self, request, context):
        group_document = spm.read_group(request.project, request.group)

        return spm_pb2.GroupDocument(group=request.group, group_document=group_document)

    @validated_request(access_level=AccessLevel.READ, validation_object_type=ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetProject(self, request, context):
        project_document = spm.read_project(request.project)

        return spm_pb2.ProjectDocument(project=request.project, project_document=project_document)

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def DeleteScenario(self, request, context):
        spm.delete_scenario(request.project, request.scenario)

        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def DeleteSystem(self, request, context):
        spm.delete_system(request.system)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def DeleteUser(self, request, context):
        spm.delete_user(request.user)

        return spm_pb2.Empty()


    @standard_error_handling()
    def CompleteUserSignup(self, request, context):


        fire.complete_user_signup(request)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT)
    @standard_error_handling()
    def ListenScenario(self, request, context):
        for scenario_document_string in spm.listen_scenario(request.project, request.scenario):
            yield spm_pb2.ScenarioDocument(scenario=request.scenario, scenario_document=scenario_document_string)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def SetScenarioProgress(self, request, context):

        try:
            spm.set_scenario_progress(request.project, request.scenario, request.job_id, request.message, request.status, request.clean, request.progress)
        except firebase.JobError as je:
            self.set_error(firebase.JobError, code=grpc.StatusCode.INTERNAL, msg=je.__str__())

        except fire.ScenarioNotFound as snf:

            self.set_error(snf.__class__, grpc.StatusCode.NOT_FOUND, msg=str(snf))

        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def SetScenarioDataTags(self, request, context):

        spm.set_scenario_metadata(scenario=request.scenario,
                                         offset=0,
                                         timezone="UTC",
                                         epoch_type="s",
                                         tags = [{'name': t} for t in request.tags],
                                         aliases ={}
                                        )

        #fire.set_scenario_data_available(request.project, request.scenario, True)
        #spm.set_data_tags(request.project, request.scenario, request.tags)
        #return spm_pb2.Scenario(scenario=request.scenario)


    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def GetScenarioDataStats(self, request, context):

        if request.execution is None or request.execution=="":
            log.debug('Getting latest exe')
            request.execution = fire.get_latest_main_execution(request.project, request.scenario)

        if not hasattr(request, 'tag') or request.tag == "" or request.tag is None:
            tag = "_index"
        else:
            tag = request.tag

        log.debug('Stat tag: '+str(tag))

        stats = spm.read_data_stats(request.scenario, request.execution, tag=tag)
        return spm_pb2.ScenarioStats(project=request.project, scenario=request.scenario, execution=request.execution,
            min = stats['min'], max = stats['max'], equi_space = stats['equi_space'],
            spacing = stats['spacing'], n_blocks =stats['n_blocks'],
            equi_block_len = stats['equi_block_len'],
            block_len0 = stats['block_len0'],
            block_len_last = stats['block_len_last'], total_val_len = stats['total_val_len'],
        )
        #return spm_pb2.Json(json=json.dumps(stats))

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def PushScenarioError(self, request, context):
        try:
            fire.push_traceback(request.project, request.scenario, request.error, request.spm_job_id)
        except fire.ScenarioNotFound as snf:

            self.set_error(snf.__class__, grpc.StatusCode.NOT_FOUND, msg=str(snf))
        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def SetScenarioJobImage(self, request, context):
        fire.set_scenario_job_image(request.project_id, request.scenario_id, request.job_id, request.name, request.path)

        return spm_pb2.ExecutionId(execution_id=None)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def PushExecutionLogEntries(self, request, context):

        try:
            spm.push_logs(request.execution_id, request.log_entries, request.timestamps)
        except IndexError:
            self.set_error(IndexError, grpc.StatusCode.INVALID_ARGUMENT, "logs and timestamps must have same length.")
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def ReadExecutionLogEntries(self, request, context):
        for l, t in spm.read_logs_timerange(request.execution_id, request.start, request.end):

            yield spm_pb2.ExeLogEntry(execution_id=request.execution_id, log_entry=l, timestamp=t)


    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def GetSignedURLs(self, request, context):

        return spm_pb2.FileSignedUrls(files=[spm_pb2.FileSignedUrl(**f) for f in spm.get_files_signed_urls(request.paths)])

    @standard_error_handling()
    def GetPublicLink(self, request, context):
        url = fire.get_url_public_link(request.id)

        return spm_pb2.SignedUrl(url=url)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO, use_user_token=True)
    @standard_error_handling()
    def GenerateScenarioUploadSignedURL(self, request: spm_pb2.FileSignedUrl, context):
        if request.content_type is None:
            request.content_type = "text/html"
        upload_url = spm.generate_resumable_upload_url(request.project, request.scenario, request.path, request.file_id, content_type=request.content_type)
        return spm_pb2.ScenarioUploadURL(**upload_url)

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def GetDataSetClosed(self, request, context):
        if request.execution is None or request.execution == "":
            log.debug('Getting latest exe')
            request.execution = fire.get_latest_main_execution(request.project, request.scenario)

        log.debug('Closed')
        return spm_pb2.Closed(is_closed=spm.get_scenario_execution_data_closed(request.scenario, request.execution))

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def GetScenarioMetaData(self, request, context):
        if request.execution is None or request.execution=="":
            log.debug('Getting latest exe')
            exe = fire.get_latest_main_execution(request.project, request.scenario)
            if exe is None or exe=="":
                return spm_pb2.Empty()
            request.execution=exe

        meta = spm.get_scenario_metadata(request.scenario, request.execution)
        #log.debug('meta type: '+ str(type(meta)))
        if meta is not None:
            def g(tag, key, def_):
                if key in tag:
                    return tag[key]
                elif def_ is not None:
                    return def_
                else:
                    raise KeyError(f'No default for key: {key}')

            def wrap_tag(tag):

                kwargs = {k: g(tag,k,d) for k, d in {
                    'name': None,
                    'displayName': '',
                    'unit': '',
                    'description': '',
                    'type': 'double',
                    'scaling': 1,
                    'offset': 0
                }.items()}

                return spm_pb2.Tag(
                    **kwargs
                )

            return spm_pb2.ScenarioMetaData(
                project=request.project,
                scenario=request.scenario,
                execution=request.execution,
                tags=[wrap_tag(tag) for tag in meta['tags']] if 'tags' in meta else [],
                aliases=[spm_pb2.Alias(tag=k, aliases=v) for k,v in meta['aliases'].items()] if 'aliases' in meta else [],
                offset=meta['offset'] if 'offset' in meta else 0,
                timezone=meta['timezone'] if 'timezone' in meta else "UTC",
                epoch_type=meta['epoch_type'] if 'epoch_type' in meta else 's'
                                )
        else:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, f"No meta data found for scenario {request.scenario}")
            return spm_pb2.Empty()


    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO, use_user_token=True)
    @standard_error_handling()
    def SetScenarioMetaData(self, request, context):

        spm.set_scenario_metadata(scenario=request.scenario,
                                  execution=request.execution,
                                         offset=request.offset,
                                         timezone=request.timezone,
                                         epoch_type=request.epoch_type,
                                         tags=request.tags,
                                         aliases=request.aliases)

        if request.project != "" and request.project is not None:
            fire.set_scenario_data_available(request.project, request.scenario, True)

        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetScenarioCustomMetaData(self, request, context):

        meta = json.dumps(spm.get_scenario_custom_metadata(request.scenario, request.execution, request.key))
        if meta is None:
            response = spm_pb2.ScenarioCustomMetaData(scenario=request.scenario, execution=request.execution,  key=request.key, meta="")
        else:
            response = spm_pb2.ScenarioCustomMetaData(scenario=request.scenario, execution=request.execution, key=request.key, meta=meta)
        return response

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def SetScenarioCustomMetaData(self, request, context):

        spm.set_scenario_custom_metadata(request.scenario, request.execution, request.key, request.meta)
        return spm_pb2.Scenario(scenario=request.scenario,  execution=request.execution)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO, use_user_token=True)
    @standard_error_handling()
    def CloseData(self, request, context):
        log.debug('finalize: '+str(request.finalized))
        spm.close_data(request.scenario, request.execution,  request.eta, request.finalized)

        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetModel(self, request, context):

        model, urls = spm.get_model(request.model_id, request.project_id, request.scenario_id)
        return spm_pb2.Model(model_id=request.model_id, model=model, files=urls)

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def SetScenarioResults(self, request, context):
        spm.set_results(request.project, request.scenario, request.names, request.values, request.units)

        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT)
    @standard_error_handling()
    def GetScenarioResults(self, request, context):
        results = spm.get_results(request.project, request.scenario)

        return spm_pb2.ScenarioResults(scenario=request.scenario, names=results['names'], values=results['values'], units=results['units'])

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def ClearScenarioResults(self, request, context):
        spm.clear_results(request.project, request.scenario)

        return spm_pb2.Scenario(scenario=request.scenario)

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def StoreConfiguration(self, request, context):
        fire.store_configuration(request.name, request.description, request.datetime_, request.user, request.configuration, request.comment, request.tags)
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ANY)
    @standard_error_handling()
    def ReadConfiguration(self, request, context):

        return spm_pb2.Configuration(fire.read_configuration(request.name))

    @standard_error_handling()
    #@validated_request(access_level=AccessLevel.WRITE)
    def PublishSubscriptionMessage(self, request, context):
        spm.publish_messages([request.channel], [request.message])
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT)
    @standard_error_handling()
    def SubscribeForUpdates(self, request, context):
        #create an event that can be used to cancel the redis stream when this stream is done.
        terminate_ = threading.Event()
        def on_rpc_done():
           log.debug('Terminate subscription')
           terminate_.set()

        message_generator = spm.subscribe_channels(request.channel_patterns, terminate_)

        context.add_callback(on_rpc_done)

        for message in message_generator:
            yield spm_pb2.SubscriptionMessage(channel=message['channel'].decode('UTF8'), message=message['data'].decode('UTF8'))

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO, custom_project_id_path='execution.project_id', custom_scenario_id_path='execution.scenario_id')
    @standard_error_handling()
    def CompleteExecution(self, request, context):
        log.debug(f'Completing execution')
        # Endpoint for instance to complete execution
        job = fire.get_job(request.execution.project_id, request.execution.scenario_id, request.execution.job_id)
        log.debug(f'Completing execution pt2 : {job is None}')
        if job is not None:
            log.debug('Job found')
            active_exe = job['active_execution'] if 'active_execution' in job else None
            log.debug(f"requested exe: {request.execution.execution_id}")
            if active_exe == request.execution.execution_id:
                log.debug('same exe')
                exe = fire.get_execution(active_exe)
                if 'instance' not in exe or exe['instance'] is None or exe['instance'] == dict(context.invocation_metadata())['instance']:
                    if request.hibernate:
                        log.debug(f"Hibernating job on completion: {exe['execution']} | {active_exe}")
                        fire.update_execution({
                            'active': True, 'hibernating': True, 'execution': exe['execution'], 'instance': None,
                            'timed_out_epoch': datetime.datetime.utcnow()
                        })

                    # Otherwise, complete execution
                    else:
                        fire.complete_execution(active_exe)
                        fire.clear_active_exe(request.execution.project_id, request.execution.scenario_id, request.execution.job_id)

        spm.publish_messages(['.'.join(['EVENT',request.execution.project_id, request.execution.scenario_id, request.execution.job_id])], [json.dumps({
            'event': 'trigger_non_main', 'project': request.execution.project_id, 'scenario': request.execution.scenario_id, 'job': request.execution.job_id
        })])

        closed = not request.hibernate

        spm.set_scenario_execution_data_closed(request.execution.scenario_id, request.execution.execution_id, closed)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.WRITE, validation_object_type = ValidationObjectType.SCENARIO)
    @standard_error_handling()
    def CompleteExecutionIgnoreInstance(self, request, context):
        spm.complete_exe_ignore_instance(request.project_id, request.scenario_id, request.job_id, request.execution_id)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def ListenExecutions(self, request, context):

        for doc in fire.listen_executions():
            yield spm_pb2.Json(json=json.dumps(doc, default=json_serial))

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def UpdateExecution(self, request, context):
        execution = json.loads(request.json)

        spm.update_execution(execution)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def ClearExecutionMemory(self, request, context):


        spm.clear_data(scenario=request.scenario, execution=request.execution, only_in_memory=True)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ADMIN)
    @standard_error_handling()
    def RemoveExecutionData(self, request, context):

        spm.clear_data(scenario=request.scenario, execution=request.execution, only_in_memory=False)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.READ, validation_object_type = ValidationObjectType.PROJECT, use_user_token=True)
    @standard_error_handling()
    def GetLatestMainExecution(self, request, context):

        exe_id = fire.get_latest_main_execution(request.project, request.scenario)

        return spm_pb2.ExecutionId(execution_id=exe_id)


deployment_client = Deploy()


class JobManagerServicer(spm_pb2_grpc.JobManagerServicer, ErrorHandler):

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def StartJob(self, request, context):
        spm.start_job(
            project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id,
            server_address=my_own_url, user_id=request.user_id, port=my_own_external_port,
            organization_id=org, resumed=False, secure_channel=my_own_secure
        )

        return spm_pb2.Job(
            project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id,
            user_id=request.user_id, organization_id=request.organization_id
        )

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def TerminateJob(self, request, context):
        # Publish command for job to terminate
        channel = _get_job_channel(request.project_id, request.scenario_id, request.job_id)
        log.debug(f'Terminating channel {channel}')

        # Mark for removal in 30 s
        job = fire.get_job(request.project_id, request.scenario_id, request.job_id)

        if 'active_execution' in job:
            execution = job['active_execution']
            exe = fire.get_execution(execution)

            hibernating = exe.get('hibernating', False)
            if hibernating:
                fire.complete_execution(exe)
                fire.clear_active_exe(request.project_id, request.scenario_id,
                                      request.job_id)
                fire.submit_progress(request.project_id, request.scenario_id,
                                     request.job_id, 'stopped', 'finished', True)
                return spm_pb2.Empty()

            if execution is not None:
                fire.submit_progress(request.project_id, request.scenario_id, request.job_id, 'pending termination',
                                     'running',
                                     True)

                if 'launch_details' in exe:
                    log.debug('Started:'+str(exe['launch_details']['start_time']))


                    response = deployment_client.set_deadline_job(
                        exe['launch_details']['name'], exe['launch_details']['namespace'], exe['launch_details']['cluster'],0
                    )

                deadline_ = 30
                fire.mark_for_removal_execution(execution, deadline_)

                log.debug(f'Execution {execution} marked for deletion')

                spm.publish_messages([channel], [{'command': 'terminate'}])
                return spm_pb2.Empty()

        spm.publish_messages([channel], [{'command': 'terminate'}])
        log.debug(f'Clearing job')
        fire.submit_progress(request.project_id, request.scenario_id, request.job_id, 'cleared', 'ready', True)

        log.debug(f'No executions marked for deletion!')
        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def ResetJob(self, request, context):
        spm.reset_job(project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id)

        return spm_pb2.Job(project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id)

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def GetExecutionStatus(self, request, context):
        try:
            status = spm.get_execution_status(request.execution_id)
            return spm_pb2.Json(json=json.dumps(status))
        except deployment.JobNotFound:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'Could not process job')

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    @standard_error_handling()
    def DeleteExecution(self, request, context):
        spm.delete_execution(request.execution_id)

        return spm_pb2.Empty()

    @validated_request(access_level=AccessLevel.WRITE, use_user_token=True)
    @standard_error_handling()
    def HibernateJob(self, request, context):
        # Signal job
        channel = _get_job_channel(request.project_id, request.scenario_id, request.job_id)
        log.debug(f'Hibernating channel {channel}')
        spm.publish_messages([channel], [{'command': 'hibernate'}])

        return spm_pb2.Job(project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id)



    @validated_request(access_level=AccessLevel.WRITE, use_user_token=True)
    @standard_error_handling()
    def ResumeJob(self, request, context):
        spm.resume_job(request.project_id, request.scenario_id, request.job_id, request.user_id, my_own_url, my_own_external_port, my_own_secure)
        return spm_pb2.Job(project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id)

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def AddJobSchedule(self, request, context):
        # Get active execution
        active_execution = spm.get_active_exe(
            project_id=request.job.project_id, scenario_id=request.job.scenario_id, job_id=request.job.job_id
        )

        if active_execution is None: self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'No active execution!')

        # Handle adding job schedule to execution
        spm.add_schedule_to_execution(execution_id=active_execution, schedule=request.schedule)

        return spm_pb2.Empty()

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def UpdateJobSchedule(self, request, context):
        # Get active execution
        active_execution = spm.get_active_exe(
            project_id=request.job.project_id, scenario_id=request.job.scenario_id, job_id=request.job.job_id
        )
        if active_execution is None: self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'No active execution!')

        # Handle updating job schedule to execution
        spm.update_execution_schedule(execution_id=active_execution, schedule=request.schedule)

        return spm_pb2.Empty()

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def RemoveJobSchedule(self, request, context):
        # Get active execution
        active_execution = spm.get_active_exe(
            project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id
        )
        if active_execution is None: self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'No active execution!')

        # Handle deleting job schedule to execution
        spm.delete_execution_schedule(execution_id=active_execution)

        return spm_pb2.Empty()

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def GetJobSchedule(self, request, context):
        # Get active execution
        active_execution = spm.get_active_exe(
            project_id=request.project_id, scenario_id=request.scenario_id, job_id=request.job_id
        )
        if active_execution is None: self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'No active execution!')

        # Handle getting job schedule to execution
        schedule = spm.get_execution_schedule(execution_id=active_execution)

        if schedule is None:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'No schedule found for job!')

        return spm_pb2.Schedule(**schedule)

    # Execution scheduling
    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def AddExecutionSchedule(self, request, context):
        # Handle adding job schedule to execution
        spm.add_schedule_to_execution(
            execution_id=request.execution.execution_id, schedule=request.schedule
        )

        return spm_pb2.Empty()

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def UpdateExecutionSchedule(self, request, context):
        # Handle updating job schedule to execution
        spm.update_execution_schedule(execution_id=request.execution.execution_id, schedule=request.schedule)

        return spm_pb2.Empty()

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def RemoveExecutionSchedule(self, request, context):
        # Handle deleting job schedule to execution
        spm.delete_execution_schedule(execution_id=request.execution_id)

        return spm_pb2.Empty()

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def GetExecutionSchedule(self, request, context):
        # Handle getting job schedule to execution
        schedule = spm.get_execution_schedule(execution_id=request.execution_id)

        if schedule is None:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, 'No schedule found for job!')

        return spm_pb2.Schedule(**schedule)

    @standard_error_handling()
    @validated_request(access_level=AccessLevel.ADMIN, use_user_token=True)
    def GetClusterInfo(self, request, context):
        return spm_pb2.ClusterInfoReply(info=[spm_pb2.ClusterInfo(name=i.name, os=i.os) for i in spm.get_cluster_info().info])

class TokenManagerServicer(spm_pb2_grpc.TokenManager, ErrorHandler):
    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def CreateRefreshToken(self, request, context):
        refresh_token = None
        try:
            # Generate refresh token, and then use refresh token to generate access token
            refresh_token = token_manager.generate_refresh_token(
                project_id=request.project_id, scenario_id=request.scenario_id, admin=request.admin,
                execution_id=request.execution_id, job_id=request.job_id, user_id=request.user_id,
                organization_id=request.organization_id,
                agent=request.agent,
                purpose=request.purpose, access_level=request.access_level
            )
        except token_manager.ValidationException as e:
            self.set_error(token_manager.ValidationException, grpc.StatusCode.UNAUTHENTICATED, e.__str__())

        except KeyError as e:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, e.__str__())

        logging.warning(f"REFRESH: {refresh_token}")

        return spm_pb2.Token(val=refresh_token)

    @standard_error_handling()
    def LoginGetRefreshToken(self, request, context):
        try:
            login = firebase.auth().sign_in_with_email_and_password(request.email, request.password)
        except requests.exceptions.HTTPError:
            self.set_error(token_manager.ValidationException, grpc.StatusCode.UNAUTHENTICATED, "Invalid credentials")
        
        if not token_manager.validate_token_request(request.token_request, login["localId"]):
            self.set_error(token_manager.TokenRequestException, grpc.StatusCode.UNAUTHENTICATED, "Insufficient privileges")
        else:
            return spm_pb2.Token(val=token_manager.create_refresh_token(request.token_request, login["localId"]))

    @validated_request(access_level=AccessLevel.DEVELOPER)
    @standard_error_handling()
    def CreateRefreshToken(self, request, context):
        return spm_pb2.Token(val=token_manager.create_refresh_token(request))

    @standard_error_handling()
    def GetAccessToken(self, request, context):
        access_token = None
        try:

            access_token = token_manager.generate_access_token(request.refresh_token.val, request.instance_id, request.project_id, request.scenario_id, request.job_id, request.execution_id)
        except token_manager.ValidationException as e:
            self.set_error(token_manager.ValidationException, grpc.StatusCode.UNAUTHENTICATED, e.__str__())
        except KeyError as e:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, e.__str__())
        return spm_pb2.Token(val=access_token)


def _initialize_channel(server, port, secure_channel, insecure_port):
    if secure_channel:
        cert = str.encode(os.getenv('NUMEROUS_CERT_CRT'))
        key = str.encode(os.getenv('NUMEROUS_CERT_KEY'))
        creds = grpc.ssl_server_credentials([(key, cert)])
        server.add_secure_port(f"[::]:{port}", creds)
    else:
        server.add_insecure_port(f'[::]:{port}')
    server.add_insecure_port(f'[::]:{insecure_port}')

    reflection.enable_server_reflection(['SPMServicer'], server)

    return server


if __name__ == '__main__':
    log.info(f"url={my_own_url}, external_port={my_own_external_port}, port={my_own_port}, insecure_port={my_own_insecure_port}, secure_channel={my_own_secure}")

    # create a gRPC server
    options = (
        ("grpc.keepalive_time_ms", 5000),
        ("grpc.keepalive_timeout_ms", 5000),
        ("grpc.keepalive_permit_without_calls", True),
        ("grpc.http2_max_pings_without_data", 0),
        ("grpc.http2_min_recv_ping_interval_without_data_ms", 5000),
        ("grpc.http2_max_ping_strikes", 3)
    )
    server = grpc.server(
        futures.ThreadPoolExecutor(max_workers=100),
        interceptors=[InstanceInterceptor()],
        options=options
    )

    # to add the defined class to the server
    spm_pb2_grpc.add_SPMServicer_to_server(SPMServicer(), server)
    spm_pb2_grpc.add_JobManagerServicer_to_server(JobManagerServicer(), server)
    spm_pb2_grpc.add_TokenManagerServicer_to_server(TokenManagerServicer(), server)
    spm_pb2_grpc.add_BuildManagerServicer_to_server(BuildManagerServicer(), server)
    health_pb2_grpc.add_HealthServicer_to_server(HealthServicer(), server)

    log.info(f'Starting server. Listening on port {my_own_port}. Using secure channel: {my_own_secure}')

    server = _initialize_channel(server, my_own_port, my_own_secure, my_own_insecure_port)
    server.start()
    serve_health_check_endpoint()

    # since server.start() will not block,
    # a sleep-loop is added to keep alive
    try:
        while True:
            time.sleep(86400)
    except KeyboardInterrupt:
        server.stop(0)
