import os
import json
import grpc
import logging
import logging.handlers
import threading
from urllib.parse import urlparse
from time import  time
from cryptography import x509
from cryptography.hazmat.backends import default_backend

from numerous_cert_server.cert_helper.get_cert import get_cert
from . import numerous_deployment_pb2, numerous_deployment_pb2_grpc
from numerous_api_client.headers.ValidationInterceptor import ValidationInterceptor


log = logging.getLogger('numerous_deployment_client')
log.setLevel(logging.DEBUG)


org = os.environ['NUMEROUS_ORGANIZATION']

class NumerousDeploymentGrpcClient:

    def __init__(self, url):
        parsed_url = urlparse(url)
        self._access_token = None
        self._refresh_token = os.getenv('NUMEROUS_DEPLOYMENT_API_REFRESH_TOKEN')
        self._prefix = org
        self.channel = self._init_channel(server=parsed_url.hostname, port=parsed_url.port)
        self.stub = numerous_deployment_pb2_grpc.DeployStub(self.channel)

        self._access_token_refresher = RepeatedFunction(
            interval=9 * 60, function=self._refresh_access_token, run_initially=True,
            refresh_token=self._refresh_token)
        self._access_token_refresher.start()

    def _init_channel(self, server, port, instance_id=None):
        cert = str.encode(get_cert(f'https://{server}:4443/cert'))
        creds = grpc.ssl_channel_credentials(cert)
        cert_decoded = x509.load_pem_x509_certificate(cert, default_backend())
        cert_cn = cert_decoded.subject.rfc4514_string().split('CN=')[-1].split(',')[0]
        options = (('grpc.ssl_target_name_override', cert_cn),)
        channel = grpc.secure_channel(f'{server}:{port}', creds, options)

        vi = ValidationInterceptor(token=self._access_token, token_callback=self._get_current_token, instance=instance_id)
        self._instance = vi.instance
        channel = grpc.intercept_channel(channel, vi)

        return channel

    def _get_current_token(self):
        return self._access_token

    def _refresh_access_token(self, refresh_token):
        token = self.stub.GetAccessToken(
            numerous_deployment_pb2.RefreshRequest(
                refresh_token=numerous_deployment_pb2.Token(val=refresh_token), prefix=self._prefix
            )
        )
        self._access_token = token.val

    def close(self):
        self._access_token_refresher.stop()


class Deploy(NumerousDeploymentGrpcClient):

    def __init__(self):
        super().__init__(os.getenv('NUMEROUS_DEPLOYMENT_SERVER_URL'))

    def close(self):
        super().close()

    def create_job(self, **spec):
        #log.debug(spec)
        reply = json.loads(self.stub.CreateJob(numerous_deployment_pb2.CreateJobRequest(prefix=org, json_spec=json.dumps(spec))).json_reply)
        return reply['name'], reply['namespace']

    def suspend_job(self, name, namespace, cluster):
        self.stub.SuspendJob(numerous_deployment_pb2.SuspendJobRequest(prefix=org, json_spec=json.dumps({
            'cluster': cluster,
            'name': name,
            'namespace': namespace,
        })))

    def delete_job(self, name, namespace, cluster):
        self.stub.DeleteJob(numerous_deployment_pb2.DeleteJobRequest(prefix=org, json_spec=json.dumps({
            'cluster': cluster,
            'name': name,
            'namespace': namespace,
        })))

    def set_deadline_job(self, name, namespace, cluster, deadline):
        self.stub.SetDeadlineJob(numerous_deployment_pb2.SuspendJobRequest(prefix=org, json_spec=json.dumps({
            'cluster': cluster,
            'name': name,
            'namespace': namespace,
            'deadline': deadline
        })))

    def get_job_status(self, name, namespace, cluster):
        return json.loads(self.stub.GetStatusJob(numerous_deployment_pb2.GetStatusJobRequest(
            prefix= org,
            json_spec=json.dumps({
            'cluster': cluster,
            'name': name,
            'namespace': namespace
        }))).json_reply)


    def get_cluster_info(self):
        return self.stub.GetClusterInfo(numerous_deployment_pb2.ClusterInfoRequest(prefix=org))


class RepeatedFunction:
    def __init__(self, interval, function, run_initially=False, *args, **kwargs):
        self._timer = None
        self.interval = interval
        self.function = function
        self.args = args
        self.kwargs = kwargs
        self.is_running = False
        self.next_call = time()

        if run_initially:
            self.function(*self.args, **self.kwargs)

    def _run(self):
        self.is_running = False
        self.start()
        self.function(*self.args, **self.kwargs)

    def start(self):
        if not self.is_running:
            self.next_call += self.interval
            self._timer = threading.Timer(self.next_call - time(), self._run)
            self._timer.start()
            self.is_running = True

    def stop(self):
        self._timer.cancel()
        self.is_running = False
