import logging
import firebase_admin
from google.oauth2 import service_account
from google.auth.transport.requests import Request

from firebase_admin import firestore, auth
import json
import os
from queue import Queue
import threading
from datetime import datetime, timedelta, timezone
from .signed_url import generate_signed_url, generate_resumable_upload
from google.cloud import storage
from google.api_core.exceptions import InvalidArgument, NotFound
from google.cloud.firestore_v1.field_path import FieldPath
from google.cloud.firestore_v1.document import DocumentReference


log = logging.getLogger('numerous_api.spm.firebase')
log.setLevel(logging.DEBUG)


bucket_url = os.getenv('NUMEROUS_BUCKET')


firebase_service_account_env = 'NUMEROUS_FIREBASE_SERVICE_ACCOUNT'
firebase_service_account = os.getenv(firebase_service_account_env)
if firebase_service_account is None:
    raise KeyError('No firebase service account in env:'+firebase_service_account_env)
else:
    service_account_data = json.loads(firebase_service_account)


storage_service_account_env = 'NUMEROUS_STORAGE_SERVICE_ACCOUNT'
storage_service_account = os.getenv(storage_service_account_env)
if storage_service_account is None:
    raise KeyError('No storage service account in env:'+storage_service_account)
else:
    service_account_data_storage = json.loads(storage_service_account)


def _get_credentials():
    credentials = service_account.Credentials.from_service_account_info(service_account_data_storage, scopes=[
        'https://www.googleapis.com/auth/cloud-platform',
        'https://www.googleapis.com/auth/userinfo.email'
    ])

    request = Request()
    credentials.refresh(request)

    return credentials


try:
    fire_app = firebase_admin.get_app()#app_name)
except ValueError as e:
    creds = firebase_admin.credentials.Certificate(service_account_data)
    fire_app = firebase_admin.initialize_app(creds)#, name=app_name)

firebase = firestore.client(app=fire_app)

try:
    #Hackish connection test!
    doc = firebase.document('/')
    doc.get()
except InvalidArgument:
    pass
except:
    log.error('Could not connect to firebase!')
    raise

class JobError(Exception):
    pass

class ScenarioNotFound(Exception):
    pass

#TODO only have standard env for google

storage_credentials = _get_credentials()

storage_client = storage.Client(credentials=storage_credentials)

def generate_resumable_upload_with_client(path, content_type="text/html"):
    return generate_resumable_upload(storage_client, bucket_url, path, content_type=content_type)


def doc_ref(path, check_exist=False):
    ref = firebase.document(path)
    doc = ref.get()
    if check_exist and not doc.get().exists:
        raise KeyError(f'Document {path} not found!')
    return ref

def __get_document_from_path(path, as_dict=True):
    # Simply returns values of a document as a dict, based on a path
    doc_ = doc_ref(path).get()

    return doc_.to_dict() if as_dict else doc_


def check_project_exist(project_id):
    if project_id == "":
        raise NotFound('Project id can not be empty')
    ref = firebase.document('Projects/'+project_id)
    doc = ref.get()
    return doc.exists

def check_scenario_exist(project_id, scenario_id):
    ref = firebase.document('Projects/'+project_id+'/Scenarios/'+scenario_id)
    doc = ref.get()
    return doc.exists

def check_job_exist(project_id, scenario_id, job_id):
    ref = firebase.document('Projects/'+project_id+'/Scenarios/'+scenario_id)
    doc = ref.get(['jobs'])
    return doc.exists and 'jobs' in (dd:=doc.to_dict()) and job_id in dd['jobs']

def submit_progress(project_id, scenario_id, spm_job_id, message=None, status=None, clean=False, progress=None):

    try:
        path = f'Projects/{project_id}/Scenarios/{scenario_id}'
        progress_ref = doc_ref(path)

        if status is None and message is None:
            raise ValueError("Neither message nor status were set when attempting to submit progress.")

        update_ = {f'jobs.{spm_job_id}.status.status': status,
                             f'jobs.{spm_job_id}.status.message': message
                             }
        if progress is not None:
            update_.update({f'jobs.{spm_job_id}.status.progress': progress})

        log.debug('prog: '+str(update_))

        progress_ref.update(update_)
    except NotFound:
        log.debug('Setting progress on non existing scenario')


def submit_execution(project_id, scenario_id, spm_job_id, execution_id, launch_details=None, instance_id=None, schedule_key='schedule_key', schedule_data=None):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'
    scen_ref = doc_ref(path)

    scen = scen_ref.get([f'inputScenarios', f'jobs']).to_dict()
    input_scenarios= scen['inputScenarios']
    is_main = scen['jobs'][spm_job_id]['isMain'] if 'isMain' in scen['jobs'][spm_job_id] else False

    try:
        execution_garbage = scen['jobs'][spm_job_id]['execution_garbage']
        log.warning('Execution Garbage: '+str(execution_garbage))
        for eg in execution_garbage:
            update_execution({'execution': eg, 'remove_deadline': datetime.utcnow()})
    except Exception as e:
        log.warning('execution garbage issue '+str(e))

    for input_scenario in input_scenarios:
        path = f'Projects/{input_scenario["projectID"]}/Scenarios/{input_scenario["scenarioID"]}'
        input_scen_exe = doc_ref(path).get(['lastest_main_execution']).to_dict()

        if not input_scen_exe is None and 'lastest_main_execution' in input_scen_exe:
            input_scenario["executionID"] = input_scen_exe['lastest_main_execution']
        else:
            input_scenario["executionID"] = ""

    scen_update = {
        f'jobs.{spm_job_id}.active_execution': execution_id,
        f'jobs.{spm_job_id}.most_recent_execution': {'execution': execution_id, 'started': datetime.utcnow()},
        f'jobs.{spm_job_id}.execution_history': firestore.ArrayUnion([{'execution': execution_id, 'started': datetime.utcnow()}]),
        f'jobs.{spm_job_id}.execution_garbage': [execution_id],
        'inputScenarios': input_scenarios,
    }

    if is_main:
        scen_update.update({
            f'data_available': False
        })
    scen_ref.update(scen_update)

    log.debug('scen updated gc')
    if is_main:
        scen_ref.update({
            f'lastest_main_execution': execution_id
        })

    path = f'Executions/{execution_id}'
    exe_ref = doc_ref(path)
    details = {
        'execution': execution_id,
        'project': project_id,
        'scenario': scenario_id,
        'job': spm_job_id,
        'instance': instance_id,
        'active': True,
        'archived': False,
        'requested_deadline': (datetime.utcnow() + timedelta(seconds=10)),
        'hibernating': False,
        schedule_key: schedule_data
    }

    logging.warning(f"DETAILS: {details}")
    if launch_details is not None:
        details.update({'launch_details': launch_details})
    exe_ref.set(details, merge=True)

def set_scenario_data_available(project_id, scenario_id, avail):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'
    scen_ref = doc_ref(path)
    scen_ref.update({f'data_available': avail})

def get_latest_main_execution(project_id, scenario_id):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'
    scen_ref = doc_ref(path)
    scen = scen_ref.get([f'lastest_main_execution']).to_dict()

    return scen['lastest_main_execution'] if 'lastest_main_execution' in scen else None

def set_execution_instance(execution_id, instance_id):
    try:
        path = f'Executions/{execution_id}'
        exe_ref = doc_ref(path)
        exe_ref.update({
            'instance': instance_id,
        })

    except NotFound:
        log.debug('Setting execution on non existing scenario')


def update_execution(execution):
    try:

        path = f'Executions/{execution["execution"]}'
        exe_ref = doc_ref(path)
        exe_ref.update(execution)

    except NotFound:
        log.debug('Setting execution on non existing scenario')


def get_execution(execution_id):

    try:

        path = f'Executions/{execution_id}'
        return doc_ref(path).get().to_dict()

    except NotFound:
        log.debug('Setting execution on non existing scenario')


def complete_execution(execution_id):
    try:

        path = f'Executions/{execution_id}'
        return doc_ref(path).update({
            'active': False, 'archiving_deadline': (datetime.utcnow()+timedelta(hours=72)),
            'forceful_termination_deadline': -1, 'hibernating': False
        })

    except NotFound:
        log.debug('Setting execution on non existing scenario')


def delete_execution(execution_id: str) -> None:
    """Will not raise an error if execution is not found."""
    try:
        exe = get_document(path=f'Executions/{execution_id}')
        exe.delete()
    except NotFound:
        log.debug('Could not find execution to delete')


def clear_active_exe(project_id, scenario_id, spm_job_id):
    try:
        path = f'Projects/{project_id}/Scenarios/{scenario_id}'
        scen_ref = doc_ref(path)

        scen_ref.update({f'jobs.{spm_job_id}.active_execution': None})

        execution_path = f'Executions/{spm_job_id}'
        exe_ref = doc_ref(execution_path)

        exe_ref.update({f'clean_deadline': datetime.utcnow()})


    except NotFound:
        log.debug('Getting non existing scenario')

def listen_executions():
    exe_queue = Queue()
    # Create a callback on_snapshot function to capture changes
    def on_snapshot(col_snapshot, changes, read_time):
        docs = []
        for doc in col_snapshot:
            try:
                docs.append(doc.to_dict())
            except Exception as e:
                logging.warning(f"DOC: {doc.id}, '{e}'")

        exe_queue.put({'docs': docs})

    col_query = firebase.collection('Executions').where('archived', '==', False)

    # Watch the collection query
    query_watch = col_query.on_snapshot(on_snapshot)

    while True:
        yield exe_queue.get()


def mark_for_removal_execution(execution_id, deadline):

    try:
        log.debug(f'Execution {execution_id} set for removal in {deadline} s')
        path = f'Executions/{execution_id}'
        exe_ref = doc_ref(path)
        exe_ref.update({
            'forceful_termination_deadline': (datetime.utcnow() + timedelta(seconds=deadline))
        })

    except NotFound:
        log.debug('Setting removal on non existing scenario')

def get_job(project_id, scenario_id, spm_job_id):

    try:
        path = f'Projects/{project_id}/Scenarios/{scenario_id}'
        scen_ref = doc_ref(path)
        doc = scen_ref.get([f'jobs']).to_dict()
        if doc is not None and 'jobs' in doc and doc['jobs']is not None and spm_job_id in doc['jobs']:
            active_job = doc['jobs'][spm_job_id]

            return active_job

    except NotFound:
        log.debug('Getting non existing scenario')


def get_job_schedule(project_id: str, scenario_id: str, job_id: str, schedule_key: str = 'schedule') -> dict or None:
    """Returns the schedule of a job, otherwise None."""
    job = get_job(project_id=project_id, scenario_id=scenario_id, spm_job_id=job_id)
    return job[schedule_key] if schedule_key in job.keys() else None


def set_results(project_id, scenario_id, result_names, values, units):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'
    result_ref = firebase.document(path)
    results = {'results': [
        {'colName': rn, 'value': v, 'unit': u} for rn, v, u in zip(result_names, values, units)
        ]
    }

    result_ref.set(results, merge=True)

def get_results(project_id, scenario_id):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'
    result_ref = firebase.document(path)
    return result_ref.get(field_paths={'results'}).to_dict().get('results')


def get_scenario_keys(project_id):
    path = f'Projects/{project_id}'
    result_ref = firebase.document(path)
    return result_ref.get(field_paths={'scenarios'}).to_dict().get('scenarios')

def clear_results(project_id, scenario_id):
    log.debug('clearing results for scenario '+scenario_id)
    set_results(project_id, scenario_id, [], [], [])

def generate_urls(files_list):
    surls = []
    for i in range(len(files_list)):
        f = files_list[i]
        surl = generate_signed_url(storage_credentials, bucket_url, f['path'], expiration=604800)
        f['url'] = surl

def generate_url(path):
    return generate_signed_url(storage_credentials, bucket_url, path, expiration=604800)

def get_scenario_files_signed_urls(scenario_document):
    file_data = []

    def get_file_val(component, key):
        if component.get(key):
            for p in component[key]:
                if p.get('type') == 'file':
                    if type(p['value']) == dict and 'path' in p['value']:
                        file_data.append(p['value'])
                    elif p['value'] is not None:
                        # else:
                        files.append(p['value'])
                    # files.append(p['value'])

    if 'simComponents' in scenario_document:
        for comp in scenario_document['simComponents']:
            get_file_val(comp, 'parameters')
            get_file_val(comp, 'inputVariables')

    generate_urls(file_data)
    return file_data


def read_scenario(project_id, scenario_id):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'
    scenario_data = __get_document_from_path(path)
    files = get_scenario_files_signed_urls(scenario_data)
    return scenario_data, files

def read_group(project_id, group_id):
    path = f'Projects/{project_id}/Groups/{group_id}'
    group_data = __get_document_from_path(path)

    return group_data

def read_project(project_id):
    path = f'Projects/{project_id}'
    proj_data = __get_document_from_path(path)

    return proj_data

def delete_scenario(project_id, scenario_id):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'

    scenario_doc_ref = firebase.document(path)

    path_group = f'Projects/{project_id}/Groups/{scenario_doc_ref.get(["groupID"]).to_dict()["groupID"]}'

    group_doc_ref = firebase.document(path_group)

    group_doc_ref.update({
        "scenarios": firestore.ArrayRemove(
            [scenario_id])
    })

    scenario_doc_ref.delete()

def delete_system(system_id):
    path = f'SimulationModels/{system_id}'
    firebase.document(path).delete()

def delete_user(userid):
    log.warning('Deleting user: '+ str(userid))
    #log.debug(str([u.email +', ' + u.uid for u in auth.list_users().users]))
    #user = auth.get_user_by_email(user_email)
    auth.delete_user(userid)
    firebase.document(f"users/{userid}").update({'deleted':True, 'deletion_time': datetime.utcnow()})

def get_url_public_link(link_id):
    link_doc_ref = firebase.document(f"Links/{link_id}")
    link_data = link_doc_ref.get()
    if not link_data.exists:
        raise NotFound('Link document not found!')

    link_data_dict = link_data.to_dict()
    return generate_url(link_data_dict['file']['report_path'])

def complete_user_signup(signup_data):
    invitation_doc_ref = firebase.document(f"invitations/{signup_data.invitationID}")
    if not (invitation_data:=invitation_doc_ref.get()).exists:
        raise NotFound('Invitation document not found!')

    invitation_data = invitation_data.to_dict()
    full_name = f"{signup_data.firstName} {signup_data.lastName}"
    log.debug(f"Creating user with email: {signup_data.email}")
    user_doc = firebase.collection("users").document()
    user_doc.set({
        'firstName': signup_data.firstName,
        'lastName': signup_data.lastName,
        'fullName': full_name,
        'mail': signup_data.email,
        'userRole': invitation_data['userRole'],
        'organisation': invitation_data['organisation'],
        'organisations': [invitation_data['organisation']]
    })
    auth.create_user(uid=user_doc.id, email=signup_data.email, display_name=full_name, password=signup_data.password)
    invitation_doc_ref.delete()
    log.debug(f"User signup for {signup_data.email} complete")


def get_user(user_id):
    return firebase.document(f'users/{user_id}').get().to_dict()


def listen_scenario(project_id, scenario_id):
    path = f'Projects/{project_id}/Scenarios/{scenario_id}'

    queue = Queue()
    # Create an Event for notifying main thread.
    callback_done = threading.Event()

    # Create a callback on_snapshot function to capture changes
    def on_snapshot(doc_snapshot, changes, read_time):
        for doc in doc_snapshot:

            queue.put(doc)

        callback_done.set()

    doc_ref = firebase.document(path)

    # Watch the document
    doc_watch = doc_ref.on_snapshot(on_snapshot)

    while True:
        doc = queue.get()
        data = doc.to_dict()
        files = get_scenario_files_signed_urls(data)
        yield data, files

def store_configuration(name, description, datetime_, user, configuration, comment, tags):
    db.collection('Configs').document().set({
        'Name': name,
        'Description': description,
        'Datetime': datetime_,
        'User': user,
        'Configuration': configuration,
        'Comment': comment,
        'Tags': tags
    })

def read_configuration(name):
    config_ref = db.collection('Configs')

    query_stream = config_ref.where('Name', '==', name).stream()

    for r in query_stream:
        return r.to_dict()


def push_traceback(project_id, scenario_id, traceback, spm_job_id):
    try:
        path = f'Projects/{project_id}/Scenarios/{scenario_id}'
        traceback_ref = firebase.document(path)

        # Create a timestamp for error in timezone GMT+2 (two hour offset from UTC)
        tz = timezone(timedelta(hours=2))
        time_stamp = datetime.now(tz=tz)

        traceback_ref.update({
            'traceback': firestore.ArrayUnion([{"timestamp": time_stamp, "error": traceback, "jobId": spm_job_id}])
        })
    except NotFound:
        log.warning('Not found -traceback')
        raise ScenarioNotFound(f'Scenario {scenario_id} not found!')


def push_formatted_error(
        project_id, scenario_id, message=None, hint=None, category=None, exception_object_type=None,
        exception_object_message=None, full_traceback=None, initializing=False
):
    try:

        path = f'Projects/{project_id}/Scenarios/{scenario_id}'
        error_ref = firebase.document(path)

        # Clear errors on init
        if initializing:
            error_ref.update({'formatted_errors': None})
            return

        # Create a timestamp for logs in timezone GMT+2 (two hour offset from UTC)
        tz = timezone(timedelta(hours=2))
        time_stamp = datetime.now(tz=tz).strftime("%Y-%d-%m %H:%M:%S")

        error_ref.update({
            'formatted_errors': firestore.ArrayUnion([{
                "timestamp": time_stamp,
                "message": message,
                "hint": hint,
                "category": category,
                "exception_object_type": str(exception_object_type),
                "exception_object_message": str(exception_object_message),
                "full_traceback": full_traceback
            }])
        })
    except NotFound:
        log.warning('Not found - form error')
        raise ScenarioNotFound(f'Scenario {scenario_id} not found!')


def get_model_files_signed_urls_(model):
    if 'Files' in model:
        files_list = list(model['Files'].values())
    else:
        files_list= []

    generate_urls(files_list)
    return files_list


def get_files_signed_urls(paths):
    files_list = [{'path': p, 'name': p.split('/')[-1]} for p in paths]

    generate_urls(files_list)
    return files_list


def check_file_exist(path):

    bucket = storage_client.bucket(bucket_url)
    return storage.Blob(bucket=bucket, name=path).exists(storage_client)


def generate_resumable_upload_url(project, scenario, file, file_id, content_type="text/html"):
    bucket_id = bucket_url
    #check path is not containing ..
    if '..' in file:
        raise PermissionError('References to parent folder is not allowed!')

    path = f'simulation_files/{scenario}/{file}'

    doc_path = f"Projects/{project}/Scenarios/{scenario}"
    filename = file.split('/')[-1]
    firebase.document(doc_path).update(
        {u'output_files.'+file_id: {'fileType': file.split('.')[-1],
                            'fileName': filename,
                            'bucket_id': bucket_id, 'report_path': path, 'timestamp': datetime.now()}})
    #firebase.document(doc_path).update({u'output_files': firestore.tf.ArrayUnion([{'fileType': file.split('.')[-1], 'fileName': file.split('/')[-1], 'bucket_id': bucket_id, 'report_path': path, 'timestamp': datetime.now()}])})


    surl = generate_resumable_upload(storage_client, bucket_id, path, content_type=content_type)

    return surl

def get_model(model_id, project_id=None, scenario_id=None):
    if project_id is None:
        path = f"SimulationModels/{model_id}"
    else:
        path = f"Projects/{project_id}/Scenarios/{scenario_id}/SimulationModel/{model_id}"
    model = __get_document_from_path(path, as_dict=True)

    return model, get_model_files_signed_urls_(model)


def set_data_tags(project_id, scenario_id, tags:list):
    path = f"Projects/{project_id}/Scenarios/{scenario_id}"

    firebase.document(path).update({u'dataTags': list(tags)})

def set_result_document(project_id, scenario_id, execution_id, results: dict):
    path = f"Projects/{project_id}/Scenarios/{scenario_id}/results/{execution_id}"

    firebase.document(path).set(results)

def get_result_document(project_id, scenario_id, execution_id):
    path = f"Projects/{project_id}/Scenarios/{scenario_id}/results/{execution_id}"

    return firebase.document(path).get().to_dict()


def set_scenario_job_image(project_id, scenario_id, job_id, image_name, image_path, repository, commit_id, internal, build):
    path = f"Projects/{project_id}/Scenarios/{scenario_id}"

    firebase.document(path).update({f'jobs.{job_id}.image.name': image_name,
                                    f'jobs.{job_id}.image.path': image_path,
                                    f'jobs.{job_id}.image.repository': repository,
                                    f'jobs.{job_id}.image.commit_id': commit_id,
                                    f'jobs.{job_id}.image.internal': internal,
                                    f'jobs.{job_id}.image.build': build,
                                    })


def set_document(path: str, contents: dict = {}, merge=False):
    get_document(path).set(contents, merge=merge)


def get_document(path: str):
    return firebase.document(path)


def get_collection(path: str):
    return firebase.collection(path)


def get_document_as_dict(path: str):
    return get_document(path).get().to_dict()


def get_collection_as_generator(path: str):
    return get_collection(path).stream()


def set_document_random_id(path: str, contents: dict = {}):
    return get_collection(path).add(contents)[1].id


def create_build(registry, repository, snapshot_id, commit_id):
    data = {
        u'repository': repository,
        u'snapshot_id': snapshot_id,
        u'commit_id': commit_id,
        u'timestamp': datetime.utcnow(),
        u'status': 'ready',
        u'message': 'ready',
        u'progress': 0
    }

    path = f'file_registries/{registry}/repository/{repository}/builds'
    # Add a new doc in collection 'cities' with ID 'LA'
    doc = firebase.collection(path).document()
    doc.set(data)
    return doc.id


def update_build(registry, repository, build_id, data_update):
    path = f'file_registries/{registry}/repository/{repository}/builds/{build_id}'
    firebase.document(path).set(data_update, merge=True)


def update_history_update(registry, repository, commit_id, update):
    path = f'file_registries/{registry}/repository/{repository}/history/{commit_id}'
    firebase.document(path).set(update, merge=True)


def get_build(registry, repository, build_id):
    path = f'file_registries/{registry}/repository/{repository}/builds/{build_id}'
    return firebase.document(path).get().to_dict()


def create_snapshot(registry, repository, snapshot_id, file_structure):
    data = {
        u'snapshot': snapshot_id,
        u'filestructure': file_structure,
        u'timestamp': datetime.utcnow()
    }

    path = f'file_registries/{registry}/repository/{repository}/snapshots/{snapshot_id}'
    # Add a new doc in collection 'cities' with ID 'LA'
    if (doc_ref:=get_document(path)).get().exists:
        raise KeyError('Already exists!')
    else:
        doc_ref.set(data)


def get_snapshot(registry, repository, snapshot_id):
    path = f'file_registries/{registry}/repository/{repository}/snapshots/{snapshot_id}'
    if (doc_ref:=get_document(path).get()).exists:
        return doc_ref.to_dict()
    else:
        raise NotFound('Commit doesnt exist')


def get_history_entry(registry, repository, commit_id):
    path = f'file_registries/{registry}/repository/{repository}/history/{commit_id}'

    if (doc_ref:=get_document(path).get()).exists:
        return doc_ref.to_dict()
    else:
        raise NotFound('History entry doesnt exist')


def get_history(registry, repository, scenario=None, history_updates=None):
    log.debug(f"Get history for {registry}/{repository} scenario={scenario} history_updates={history_updates}")
    query = firebase.collection(f'file_registries/{registry}/repository/{repository}/history')

    if scenario:
        log.debug(f"Filtering history for scenario={scenario}")
        query = query.where("scenario","==", scenario)

    if history_updates is not None and len(history_updates)>0:
        log.debug(f"Filtering history for commits={','.join(history_updates)}")
        query = query.where(FieldPath.document_id(), "in", [DocumentReference('file_registries', registry, 'repository', repository, 'history', update, client=firebase) for update in history_updates])

    return [{**doc.to_dict(), "id": doc.id} for doc in
            sorted(query.limit(100).stream(), reverse=True, key=lambda doc: doc.get('timestamp'))]


def create_history_update(registry, request, user_id):
    scenario_doc = firebase.document(f'file_registries/{registry}/repository/{request.repository}/scenarios/{request.scenario_id}')
    if not scenario_doc.get().exists:
        scenario_doc.set({'snapshots': [request.snapshot_id]})
    else:
        scenario_doc.update({'snapshots': firestore.ArrayUnion([request.snapshot_id])})

    scenario_snapshots = scenario_doc.get(field_paths={'snapshots'}).to_dict().get('snapshots')
    scenario_verison = len(scenario_snapshots) - list(reversed(scenario_snapshots)).index(request.snapshot_id) - 1

    history_update_data = {
        'snapshot': request.snapshot_id,
        'scenario': request.scenario_id,
        'project': request.project_id,
        'scenario_version': scenario_verison,
        'parent_commit_id': request.parent_commit_id,
        'comment': request.comment,
        'timestamp': datetime.now(),
        'user_id': user_id
    }

    history_update_doc = firebase.collection(f'file_registries/{registry}/repository/{request.repository}/history').document()
    history_update_doc.set(history_update_data)

    return history_update_doc.id
