import numpy as np
from numerous_api_client.client import open_client
import os

from time import time
import logging
import datetime
import traceback
from dotenv import load_dotenv
import random
from time import sleep, time

load_dotenv()
logging.basicConfig(level=logging.DEBUG)
log = logging.getLogger('main.simulation')


class Debouncer:

    def __init__(self, debounce):
        self.debounce = debounce
        self.last = -999999

    def __call__(self, f, *args, **kwargs):
        tic = time()
        if tic > self.last+self.debounce:
            self.last = tic
            f(*args, **kwargs)


def write_example_data(output):
    n_tags = 100
    tags = [f'tag_{j}' for j in range(n_tags)]

    len_data = 24*365*25
    data_series = np.arange(len_data)
    data = {f'tag_{i}': data_series for i in range(n_tags)}

    data['_index'] = np.arange(len_data)*3600

    for i in range(len_data):

        output.write_row({t: i for t in tags})
    return tags

def run(job_id=None, project=None, scenario=None, workpath="."):

    #Open the numerous client
    with open_client(job_id=job_id, project=project, scenario=scenario) as client:
        try:


            with client.open_writer() as output:

                log.debug('Writer opened!')

                tags = write_example_data(output)

                log.debug('Write completed!')


                tic = time()
                data_generator = client.data_read_raw(tags=tags)
                [r for r in data_generator]
                print('Read raw in: ', time() - tic)

                #Read data as rows
                #This function will create a generator which when iterated over will yield a single row of data - 1 value pr. tag
                #Typical usecase is when consuming each row at a time for a digital twin simulation - basically every value needs to be processed as input for a simulation
                tic = time()
                data_generator = client.read_data_as_row(tags=tags)

                count = 0
                for row in data_generator:
                    count+=1
                print('Read rows in: ', time()-tic)

                log.debug(f'read {count} rows')

                #Read data as df
                #This function will return the data in a pandas dataframe each tags as a column, rows corresponds to values.
                #typical usecase is for reporting or postprocessing to get the data as fast as possible and in a format that lends itself to aggregation and plotting
                tic = time()
                df = client.data_read_df(tags=tags)
                print('Read df in: ', time()-tic)
                #Read as stream
                #This function outputs data in "raw" database structure of blocks. A block is a single tag and maybe 1000 values. This is the fastest way to get data.
                #Typical usecase for this is in streaming a few tags in a timeseries graph - data arrives fast to update the plot fast.
                count = 0
                tic = time()
                for b in client.read_data_stream(tags=tags):
                    count +=1
                print('Read stream blocks in: ', time()-tic)

                log.debug(f'read {count} blocks')

        except (KeyboardInterrupt):
            client.set_scenario_progress('example terminated', 'terminated', force=True)

        except Exception:
            tb = traceback.format_exc()
            log.error(tb)
            client.set_scenario_progress('example failed', 'failed', force=True)

if __name__ == '__main__':
    run()