from .firestore_files import firestore_classes as fire
import json


class _ExporterBase:
    def __init__(self, out_path='./outputs', service_account='reader.json', use_emulator=False):
        self._reader = fire.FirestoreReader(service_account=service_account, use_emulator=use_emulator)
        self._out_path = out_path

        self._json = {}

    def export_documents(self, path: str, collections: list or str) -> None:
        """Add data to a collection by reading data from a document specified by the path.
        Also loop over any collections in document, if specified"""
        # Construct path of the collection the document is in.
        _col_path = "/".join(path.split('/')[:-1])
        print("EXPORTING: ", _col_path, path)

        # Add or get the data path for the collection, and add the document data.
        self._json[_col_path]['data'] = self._json[_col_path].get('data', [])
        self._json[_col_path]['data'].append(self._reader.get_document(path=path))

        # Check collections. If a list is specified -> loop over it. If a '*' is specified -> get all.
        if collections and isinstance(collections, list):
            for col in collections:
                self.export_collection(path=col.path, documents=col.documents)
        elif collections and isinstance(collections, str) and collections == '*':
            for col in self._reader.get_all_collections(path=path):
                self.export_collection(path=col.path, documents=col.documents)

    def export_collection(self, path: str, documents: list or str) -> None:
        """Create collection in the exporter _json variable and populate it with the data
        from the documents that are specified."""
        # Ensure that the json at the specified path is a dict.
        self._json[path] = self._json.get(path, {})

        # Check documents. If a list is specified -> loop over it. If a '*' is specified - get all.
        if documents and isinstance(documents, list):
            for doc in documents:
                self.export_documents(path=doc.path, collections=doc.collections)
        elif documents and isinstance(documents, str) and documents == '*':
            for doc in self._reader.get_all_documents(path=path):
                self.export_documents(path=doc.path, collections=doc.collections)

    def get_json(self) -> str:
        """Return the json data."""
        return json.dumps(self._json, default=fire.FirestoreObject.to_json)

    def save_json_to_file(self, file_name: str = 'out.json') -> None:
        """Save current exported JSON to a file."""
        with open(f"{self._out_path}/{file_name}", 'w+') as f:
            json.dump(self._json, f, default=fire.FirestoreObject.to_json)


class Exporter:
    def __init__(self, out_path='./outputs', use_emulator=False, service_account=None):
        self._exporter = _ExporterBase(out_path=out_path, use_emulator=use_emulator, service_account=service_account)

    def export_from_spec(self, spec: fire.FirestoreSpec) -> None:
        """Export data from firestore based on a spec."""
        print("-------------------------------------------------------------")
        print("|                    EXPORTING FROM SPEC                    |")
        print("-------------------------------------------------------------")
        for col in spec.collections:
            self._exporter.export_collection(path=col.path, documents=col.documents)

    def save_export_data_to_file(self, file_name: str):
        """Save the currently collected json data to a file."""
        self._exporter.save_json_to_file(file_name=file_name)

    def read_exported_json(self) -> str:
        """Read the data that is currently stored in the exporter."""
        return self._exporter.get_json()
