import json
import firebase_admin
from firebase_admin import firestore
import dataclasses
import typing
import os
import uuid
import base64

# Emulator imports
from unittest.mock import Mock
import google.auth.credentials
from google.cloud import firestore as emulator_firestore

firebase_admin.initialize_app()


# TODO: Check if two firestore instances are slow!


@dataclasses.dataclass
class FirestoreObject:
    path: str
    data: dict

    @staticmethod
    def to_json(obj):
        return obj.__dict__ if hasattr(obj, '__dict__') else str(obj)


@dataclasses.dataclass
class FirestoreDocument(FirestoreObject):
    collections: list or str
    type_: str = "DOCUMENT"


@dataclasses.dataclass
class FirestoreCollection(FirestoreObject):
    documents: list or str
    type_: str = "COLLECTION"


@dataclasses.dataclass
class FirestoreSpec:
    collections: typing.List[FirestoreCollection] = dataclasses.field(default_factory=list)

    def from_json(self, file_path: str) -> None:
        """Loads a spec from a JSON file."""
        with open(file_path, 'r') as f:
            json_data = json.loads(f.read())

        for col in json_data['collections']:
            _docs = '*' if col['documents'] == '*' else [FirestoreDocument(path=doc, data={}, collections='*') for doc in col['documents']]
            self.collections.append(FirestoreCollection(path=col['path'], data={}, documents=_docs))

    def to_json(self):
        return json.dumps(self.collections, default=FirestoreObject.to_json)


class _FirestoreEmulatorInitializer:
    def __init__(self, service_account: str, app_name: str = 'firestore_emulator'):
        self.service_account = service_account
        self.app_name = app_name

        os.environ['FIRESTORE_EMULATOR_HOST'] = 'localhost:8080'

    def initialize_client(self):
        return emulator_firestore.Client(
            project="numerous-development",
            credentials=Mock(spec=google.auth.credentials.Credentials),
        )


class _FirestoreInitializer:
    def __init__(self, service_account: str, app_name: str = 'firestore_base'):
        print('??', service_account)
        self._app_name = app_name
        self._service_account_path = service_account

        self._client = None

    def initialize_client(self) -> firestore.client:
        bootstrap_conf = json.loads(os.getenv("NUMEROUS_BOOTSTRAP_CONFIGURATION", "{}"))
        _env_service_account = bootstrap_conf.get('firebase_service_account', None)
        if _env_service_account is None:
            print("ENV service account was None. Getting from file:", self._service_account_path)
            credentials = firebase_admin.credentials.Certificate(self._service_account_path)
        else:
            print(f"Got service account from ENV.")
            _decoded = base64.b64decode(_env_service_account)
            credentials = firebase_admin.credentials.Certificate(json.loads(_decoded))

        fire_app = firebase_admin.initialize_app(credentials, name=self._app_name + '_' + str(uuid.uuid4())[:8])
        self._client = firestore.client(app=fire_app)

        return self._client


class _FirestoreBase:
    def __init__(self, service_account: str, app_name: str = 'firestore_base', use_emulator=False):
        print('?', service_account)
        # Initialize firestore client based on ENV variable
        self._initializer_class = _FirestoreEmulatorInitializer if use_emulator else _FirestoreInitializer
        self._initializer = self._initializer_class(service_account=service_account, app_name=app_name)
        self._client = self._initializer.initialize_client()

    def get_collection_stream(self, path: str):
        return self._client.collection(path[1:])

    def get_doc(self, path: str) -> dict:
        return self._client.document(path[1:]).get().to_dict()

    def get_all_collections(self, path: str) -> list:
        return list(self._client.document(path[1:]).collections())

    def get_all_documents(self, path: str) -> firestore.firestore.Client.collections:
        return self._client.collection(path[1:]).stream()

    def write_document(self, path: str, data: dict) -> None:
        self._client.document(path[1:]).set(data)

    @staticmethod
    def get_documents_from_collection(collection) -> list:
        return [document.to_dict() for document in collection.stream()]


class FirestoreReader:
    def __init__(self, service_account, use_emulator=False):
        self._firestore_client = _FirestoreBase(service_account=service_account, app_name='firestore_reader', use_emulator=use_emulator)

    @staticmethod
    def _format_firestore_path(path: str) -> str:
        """Prepend / to path."""
        return '/' + path if path[0] != '/' else path

    def get_document(self, path: str) -> FirestoreDocument:
        """Get a document as a dict from firestore and format it."""
        return FirestoreDocument(path=path, collections='*', data=self._firestore_client.get_doc(path=path))

    def get_collection(self, path: str) -> FirestoreCollection:
        """Get a collection as a dict from firestore and format it."""
        collection_stream = self._firestore_client.get_collection_stream(path=path)
        collection_documents = self._firestore_client.get_documents_from_collection(collection=collection_stream)

        return FirestoreCollection(path=path, documents='*', data={'documents': collection_documents})

    def get_all_documents(self, path: str) -> typing.List[FirestoreDocument]:
        """Get all documents from a path."""
        return [FirestoreDocument(path=f"{path}/{doc.id}", data=doc.to_dict(), collections='*') for doc in self._firestore_client.get_all_documents(path=path)]

    def get_all_collections(self, path: str) -> typing.List[FirestoreCollection]:
        """Get all collections from a path."""
        return [FirestoreCollection(path=f"{path}/{col.id}", documents="*", data={}) for col in self._firestore_client.get_all_collections(path=path)]


class FirestoreWriter:
    def __init__(self, service_account, use_emulator=False):
        self._firestore_client = _FirestoreBase(service_account=service_account, app_name='firestore_writer', use_emulator=use_emulator)

    def write_document(self, document: FirestoreDocument):
        self._firestore_client.write_document(path=document.path, data=document.data)
