from google.auth.transport.requests import Request
from google.cloud import storage
from google.oauth2 import service_account
from google_buckets.utils.zipper import Zipper
import requests
import json
import os
import typing


class _BaseClient:
    def __init__(self, bucket_url: str, service_account_name: str = './service_accounts/bucket.json', using_client=True):
        self._service_account_data = None

        if using_client:
            self._init_service_account(service_account_name=service_account_name)
            self._bucket_url = bucket_url
            self._client = storage.Client(credentials=self._get_credentials())

        self._zipper = Zipper()

    def _init_service_account(self, service_account_name: str) -> None:
        _env_service_account = os.environ.get('NUMEROUS_STORAGE_SERVICE_ACCOUNT', None)
        if _env_service_account is None:
            print("Env service account was None. Getting from file:", './google_buckets/service_accounts/' + service_account_name)
            self._service_account_path = './google_buckets/service_accounts/' + service_account_name
            self._service_account_data = self._load_service_account()
        else:
            print("Using environment service account.")
            self._service_account_data = _env_service_account

    @staticmethod
    def _format_path(path: str) -> str:
        """Add / to path."""
        return path if path[0] != '/' else path[1:]

    def _load_service_account(self) -> str:
        with open(self._service_account_path) as f:
            return json.loads(f.read())

    def _get_credentials(self) -> service_account.credentials:
        credentials = service_account.Credentials.from_service_account_info(self._service_account_data, scopes=[
            'https://www.googleapis.com/auth/cloud-platform',
            'https://www.googleapis.com/auth/userinfo.email'
        ])

        request = Request()
        credentials.refresh(request)

        return credentials

    def check_file_exist(self, path: str) -> bool:
        """Check if a file exists. Returns bool."""
        bucket = self._client.bucket(self._bucket_url)
        return storage.Blob(bucket=bucket, name=path).exists(self._client)

    def upload_file(self, local_file_path: str, bucket_path: str, override: bool = False, metadata: dict or None = None) -> None:
        """Upload a file to bucket."""
        bucket_path = self._format_path(bucket_path)

        if self.check_file_exist(path=bucket_path) and not override:
            raise FileExistsError(f"File on path: '{bucket_path}' already exists and override option was false.")

        bucket = self._client.bucket(self._bucket_url)
        blob = storage.Blob(bucket=bucket, name=bucket_path)
        blob.metadata = metadata
        blob.upload_from_filename(local_file_path)


# TODO: Rename...
class FirestoreDataClient:
    def __init__(self, bucket_url, service_account_name='bucket.json', using_client=True):
        self._data_client = _BaseClient(bucket_url=bucket_url, service_account_name=service_account_name, using_client=using_client)
        self._zipper = Zipper()
        self._data_folder = './google_buckets/tmp/'
        self._in_zip_name = '__temp_in__.zip'
        self._out_zip_name = '__temp_out__.zip'

    @staticmethod
    def _format_file_name(file_name: str) -> str:
        return file_name + '.zip' if not file_name.endswith('.zip') else file_name

    def upload_file(
            self, local_file_path: str, remote_file_name: str, metadata: dict or None = None,
            override: bool = False, keep_zip: bool = False, _tmp_zip_override_name: str or None = None
    ) -> None:
        """Zip local file and upload with upload client. Deletes zip-file afterwards."""
        remote_file_path = f'{remote_file_name}'

        _temp_file_path = './google_buckets/tmp/'
        _temp_file_name = self._format_file_name(_tmp_zip_override_name) if _tmp_zip_override_name else self._out_zip_name
        _temp_file_complete = self._data_folder + _temp_file_name

        self._zipper.zip_file(file_path=local_file_path, out_path=_temp_file_complete)
        self._data_client.upload_file(local_file_path=_temp_file_complete, bucket_path=remote_file_path, override=override, metadata=metadata)

        # Option to keep the zip file after upload
        if not keep_zip:
            self._zipper.delete_zip_file(file_path=_temp_file_complete)
        else:
            print("Kept uploaded zip. Path:", os.getcwd() + '\\' + _temp_file_complete[2:].replace('/', '\\'))

    def download_from_signed_url(self, url):
        print(url)
        response = requests.get(url, allow_redirects=True)

        _local_file_path = self._data_folder + self._in_zip_name
        print("Downloading from signed url: ", url.split('?')[0], "to: ", _local_file_path)
        with open(_local_file_path, 'wb') as f:
            f.write(response.content)
        return _local_file_path

    def clean_tmp_folder(self, post_fix: str = None):
        """
        Clean the tmp folder for zip files. Set post-fix to either 'in' or 'out' to remove in/out files specifically
        """
        if post_fix not in ['in', 'out'] and post_fix is not None:
            raise ValueError("Invalid post-fix specified!")
        for root, dirs, files in os.walk(self._data_folder):
            for file in files:
                if post_fix == 'in' and file == self._in_zip_name:
                    os.remove(root + file)
                elif post_fix == 'out' and file == self._out_zip_name:
                    os.remove(root + file)
                else:
                    os.remove(root + file)

    def get_files_from_signed_urls(self, url_list: typing.List[str], out_path: str) -> str:
        for url in url_list:
            file_path = self.download_from_signed_url(url=url)
            self._zipper.unzip_file(file_path=file_path, out_path=out_path)
        return out_path

