from zipfile import ZipFile, ZIP_DEFLATED
from os.path import basename
import os


class Zipper:
    @staticmethod
    def _format_zip_file_name(file_name: str) -> str:
        """Add a .zip to the file name, if it is not already correct."""
        if file_name.endswith('.zip'):
            return file_name
        if file_name.split('.')[-1] != '.zip':
            return ".".join(file_name.split('.')[:-1])+".zip"
        return file_name + '.zip'

    def delete_zip_file(self, file_path: str) -> None:
        """Simply deletes a file."""
        _formatted_file_path = self._format_zip_file_name(file_path)
        if os.path.exists(_formatted_file_path):
            os.remove(_formatted_file_path)
        else:
            print(f"No file found at: {_formatted_file_path}")

    def zip_file(self, file_path: str, out_path: str):
        """Zip a file at 'file_path' and store it at 'out_path'."""
        out_path = self._format_zip_file_name(file_name=out_path)
        with ZipFile(out_path, 'w') as f:
            f.write(file_path, basename(file_path), compress_type=ZIP_DEFLATED)

    def zip_folder(self, folder_path: str, out_path: str) -> None:
        """Zip a file at 'folder_path' and store it at 'out_path'."""
        out_path = self._format_zip_file_name(file_name=out_path)

        with ZipFile(out_path, 'w') as f:
            for folder_name, sub_folders, filenames in os.walk(folder_path):
                for filename in filenames:
                    file_path = os.path.join(folder_name, filename)
                    f.write(file_path, basename(file_path), compress_type=ZIP_DEFLATED)

    def unzip_file(self, file_path: str, out_path):
        """Unzips a zip file at 'file_path' to 'out_path'"""
        file_path = self._format_zip_file_name(file_path)
        print(file_path)
        with ZipFile(file_path, 'r') as zip_ref:
            zip_ref.extractall(out_path)
        os.remove(file_path)
        return out_path
