import firestore_files.firestore_classes as fire
import google_buckets.bucket_classes as bucket
import typing
import os
import json

import exporter as exp
import importer as imp


def export_data_from_spec(
        spec_path: str, json_output_file_name: str, json_output_path: str = './outputs',
        use_emulator: bool = False, override_bucket_file: bool = False
) -> None:
    """
    Exports firestore data based on a spec and uploads it to google buckets as a zip file.
    :param spec_path: Relative path to the JSON spec used to get data.
    :param json_output_file_name: The name of JSON file you would like the ouput to be saved in. Also the name of the file in buckets.
    :param json_output_path: (Optional) The folder the exported JSON will be located in.
    :param use_emulator: (Optional) Use firestore emulator. Will attempt to connect to localhost:8080.
    :param override_bucket_file: (Optional) Set True to override the current file in buckets with 'json_output_file_name' as name.
    """
    bucket_url = os.environ.get('BUCKET_URL', "bootstrapping")
    bucket_client = bucket.FirestoreDataClient(bucket_url=bucket_url, service_account_name="secret_terraform_admin.json")

    _formatted_json_file_name = json_output_file_name + '.json' if not json_output_file_name.endswith('.json') else json_output_file_name

    # Initialize a spec from a json file
    spec = fire.FirestoreSpec()
    spec.from_json(file_path=spec_path)

    # Export spec to a json file
    exporter = exp.Exporter(out_path=json_output_path, use_emulator=use_emulator)
    exporter.export_from_spec(spec=spec)
    exporter.save_export_data_to_file(file_name=_formatted_json_file_name)

    # Zip and upload json file to buckets
    _local_file_path = f"{json_output_path}/{_formatted_json_file_name}"
    bucket_client.upload_file(
        local_file_path=_local_file_path,
        remote_file_name=json_output_file_name.split('.')[-1],
        override=override_bucket_file
    )

    # Remove the json file
    os.remove(_local_file_path)


def import_data_from_bucket_urls(
        signed_url_list: typing.List[str], read_path: str = './outputs/zips', use_emulator: bool = False
) -> None:
    """
    Import firestore data from a list of bucket URLs pointing to export data generated by the exporter as seen above.
    :param signed_url_list: List of signed URLs for files in buckets.
    :param read_path: (Optional) Path to store and read zip-files from.
    :param use_emulator: (Optional) Use firestore emulator. Will attempt to connect to localhost:8080.
    """
    bucket_url = os.environ.get('BUCKET_URL', "bootstrapping")
    bucket_client = bucket.FirestoreDataClient(bucket_url=bucket_url, using_client=False)

    # Download files from bucket list
    bucket_client.get_files_from_signed_urls(url_list=signed_url_list, out_path=read_path)

    # Initialize importer and import data into firestore
    importer = imp.Importer(use_emulator=use_emulator, read_path=read_path)
    importer.load_json_from_folder()
    importer.import_from_json()

    # Delete temp files
    for root, dirs, files in os.walk(read_path):
        for file in files:
            os.remove(f"{root}/{file}")


if __name__ == '__main__':
    # export_data_from_spec(spec_path='./specs/spec.json', json_output_file_name='users', override_bucket_file=True)

    boostrap_conf = json.loads(os.getenv("NUMEROUS_BOOTSTRAP_CONFIGURATION"))
    import_data_from_bucket_urls(signed_url_list=list(boostrap_conf['firebase_bootstrap_urls']), use_emulator=False)
