import json
import logging
import os
import time
from typing import Tuple

import urllib3
from google.auth.transport.requests import Request
from google.cloud import container_v1
from google.oauth2 import service_account
from kubernetes import client
from kubernetes.client.api_client import ApiClient
from kubernetes.client.exceptions import ApiException

from yaml_templates.yaml_generator import YamlGenerator


urllib3.disable_warnings()


log = logging.getLogger('numerous_kubernetes.kubernetes')
log.setLevel(logging.DEBUG)


class JobNotFound(Exception):
    pass


# Decorator for all endpoints
def endpoint(exception_return_values: Tuple = (None, ), raise_exception=False, print_exception_message=True, custom_execption_cls=None):
    def decorator(func):
        def wrapper(self, *args, **kwargs):
            try:
                self.update_client()
                return func(self, *args, **kwargs)
            # Raise caught exception and raise either caught or custom cls
            except ApiException as exception:
                if raise_exception:
                    if custom_execption_cls is None:
                        raise exception
                    else:
                        raise custom_execption_cls
                if print_exception_message:
                    logging.warning(f"Endpoint '{func.__name__}' raised an ApiException: {exception}")
                return exception_return_values
        return wrapper
    return decorator


class MiniKubeClientFactory:
    
    def get_client(self):
        self.auto = False
        api_configuration = client.Configuration()
        api_configuration.host = "http://host.docker.internal:8000"
        client.Configuration.set_default(api_configuration)
        return ApiClient(api_configuration, header_name="Host", header_value="127.0.0.1:8000")


class GKEClientFactory:

    def __init__(self, project_id, cluster_id, cluster_zone=None, service_account={}, auto=False):
        self.service_account = service_account
        self.auto =  auto
        self._cluster_manager_client = container_v1.ClusterManagerClient(credentials=self.get_credentials())
        log.debug(f'Connecting to: {cluster_id}, with zone: {cluster_zone}, in project: {project_id} in auto-pilot mode: {self.auto}')
        self._cluster = self._cluster_manager_client.get_cluster(name= f"projects/{project_id}/locations/{cluster_zone}/clusters/{cluster_id}")

    def get_credentials(self):
        credentials = service_account.Credentials.from_service_account_info(self.service_account, scopes=[
            'https://www.googleapis.com/auth/cloud-platform',
            'https://www.googleapis.com/auth/userinfo.email'
        ])

        request = Request()
        credentials.refresh(request)

        return credentials

    def get_client(self):
        api_configuration = client.Configuration()
        api_configuration.host = "https://" + self._cluster.endpoint + ":443"
        api_configuration.verify_ssl = False
        api_configuration.api_key = {"authorization": "Bearer " + self.get_credentials().token}
        client.Configuration.set_default(api_configuration)
        return ApiClient(api_configuration)


class KubernetesClusterAPI:
    def __init__(self, os, name, client_factory):
        self.os = os
        self.name = name
        self._key_upd = {}
        self._yaml_generator = YamlGenerator()
        self.client_factory = client_factory
        self.update_client()

    def update_client(self):
        if 'update' not in self._key_upd:
            self._key_upd['update'] = -10000

        if time.time() - self._key_upd['update'] > 3000:
            self._key_upd['update'] = time.time()
            self.api_client = self.client_factory.get_client()

    def get_info(self):
        return {
            'name': self.name,
            'os': self.os
        }

    @endpoint(exception_return_values=(None, None), raise_exception=True)
    def create_job(self, name: str, image: str, namespace: str, nodepool: str, env_variables: dict,
                   cpu_limit: float, cpu_request: float, memory_limit: float, memory_request: float,
                   user_id: str, organization_id: str) -> Tuple[str, str]:
        """Creates job in namespace. Return name and namespace of created job."""
        batch_api = client.BatchV1Api(api_client=self.api_client)
        namespace = namespace.lower()

        self.create_namespace(name=namespace)

        if self.client_factory.auto:
            job_template = self._yaml_generator.generate_yaml({
                'name': name, 'image': image, 'namespace': namespace, 'replicas': 1,
                'cpu_limit': cpu_limit, 'cpu_request': cpu_request, 'memory_limit': memory_limit,
                'memory_request': memory_request,
                'os': self.os
            }, template_name='job_auto', env_variables=env_variables)
        else:
            job_template = self._yaml_generator.generate_yaml({
                'name': name, 'image': image, 'namespace': namespace, 'replicas': 1, 'nodepool': nodepool,
                'cpu_limit': cpu_limit, 'cpu_request': cpu_request, 'memory_limit': memory_limit,
                'memory_request': memory_request,
                'os': self.os
            }, template_name='job_auto', env_variables=env_variables)
        log.debug(f"Created job template:{job_template}")

        api_response = batch_api.create_namespaced_job(body=job_template, namespace=namespace)

        log.debug('Job API response: %s', api_response)
        return api_response.metadata.name, api_response.metadata.namespace

    @endpoint(exception_return_values=(None, None))
    def delete_job(self, name: str, namespace: str = "default") -> Tuple[str, str]:
        """Deletes a namespaced job. Returns name and namespace of deleted job."""
        batch_api = client.BatchV1Api(api_client=self.api_client)
        body = client.V1DeleteOptions(propagation_policy='Background')
        batch_api.delete_namespaced_job(name=name, namespace=namespace, body=body)
        return name, namespace

    @endpoint(exception_return_values=(None, None))
    def suspend_job(self, name: str, namespace: str = "default") -> Tuple[str, str]:
        """Patches a namespaced job to suspend it. Returns name and namespace of deleted job."""
        batch_api = client.BatchV1Api(api_client=self.api_client)
        job_template = {'spec': {'suspend': True}}
        log.info(f'Patch: {job_template}')
        try:
            batch_api.patch_namespaced_job(name=name, namespace=namespace, body=job_template)
        except Exception as e:
            logging.warning(f"Encountered exception when suspending job: '{e}'. Passing.")
        return name, namespace

    @endpoint(exception_return_values=(None, None))
    def set_deadline_job(self, name: str, namespace: str = "default", deadline: int = 0) -> Tuple[str, str]:
        """Sets deadline for job. Returns name and namespace of deleted job."""
        batch_api = client.BatchV1Api(api_client=self.api_client)
        job_template = {'spec': {'activeDeadlineSeconds': deadline}}
        batch_api.patch_namespaced_job(name=name, namespace=namespace, body=job_template)
        return name, namespace

    @endpoint(print_exception_message=False)
    def create_namespace(self, name: str = 'default') -> str:
        """Creates namespace. Returns name of created namespace."""
        log.info(f"Creating namespace {name}")
        core_api = client.CoreV1Api(api_client=self.api_client)
        namespace_template = self._yaml_generator.generate_yaml({'name': name}, 'namespace')
        namespace = core_api.create_namespace(namespace_template)
        return namespace

    #@endpoint(raise_exception=True, custom_execption_cls=JobNotFound())
    def get_job_status(self, name: str, namespace: str) -> dict:
        log.debug(f'Get status: {self.name}, {name}, {namespace}')
        job_status = {}
        try:
            batch_api = client.BatchV1Api(api_client=self.api_client)
            job = batch_api.read_namespaced_job_status(name, namespace)

            log.debug(f'Status: %s', job.status)

            status = job.status

            job_status = {
                'active': status.active,
                'failed': status.failed,
                'succeeded': status.succeeded,
                'start_time': status.start_time.timestamp() if status.start_time is not None else None,
                'completion_time': status.completion_time.timestamp() if status.completion_time is not None else None,
                'conditions': [{
                    'status': c.status,
                    'type': c.type,
                    'message': c.message,
                    'reason': c.reason
                } for c in status.conditions] if status.conditions is not None else []
            }
            pods = {}
            core_api = client.CoreV1Api(api_client=self.api_client)
            log.debug(f'Job name: {name}')
            list_pods = core_api.list_namespaced_pod(namespace=namespace, label_selector=f"job-name=={name}")

            for p in list_pods.items:
                try:
                    pod_logs = core_api.read_namespaced_pod_log(p.metadata.name, namespace=namespace)
                    pods[p.metadata.name] = {'logs': pod_logs, 'phase': p.status.phase}
                except ApiException as ae:
                    pods[p.metadata.name] = {'phase': p.status.phase,
                                             'exception_status': ae.status,
                                             'exception_body': ae.body,
                                             'exception_reason': ae.reason}
            job_status['pods'] = pods

        except client.exceptions.ApiException:
            log.exception("ApiException")
        return job_status

    @endpoint()
    def get_pods_execution(self, name: str, namespace: str) -> str:
        """Get pods with job name. Returns name of created namespace."""
        core_api = client.CoreV1Api(api_client=self.api_client)
        return core_api.list_namespaced_pod(namespace=namespace, label_selector=f"job-name=={name}")


deploy_conf_json = os.getenv('NUMEROUS_DEPLOYMENT_CONFIGURATIONS')
kube_configurations = json.loads(deploy_conf_json)
kube_clusters = {}

for name, kube_configuration in kube_configurations.items():
    try:
        if "MINIKUBE_DEPLOYMENT" in os.environ:
            log.debug(f"Creating minikube client factory for {name}")
            factory = MiniKubeClientFactory()
        else:
            log.debug(f"Creating GKE client factory for {name}")
            factory = GKEClientFactory(cluster_id=kube_configuration['cluster_name'],
                                       cluster_zone=kube_configuration.get('cluster_zone'),
                                       project_id=kube_configuration['service_account']['project_id'],
                                       service_account=kube_configuration['service_account'],
                                       auto = kube_configuration.get('auto-pilot', False))
        kube_clusters[name] = KubernetesClusterAPI(kube_configuration["os"], name, factory)
    except:
        log.exception(f'Error using cluster: {name}')

default_client = kube_clusters['default']
