import os
import sys
import logging
import jwt
logging.basicConfig(level=logging.INFO)
streamhandler = logging.StreamHandler(sys.stdout)

if os.getenv('NUMEROUS_LOCAL') == 'True':
    from env_from_yaml_secrets import load_env_from_yaml

    load_env_from_yaml()

from concurrent import futures
import time

import grpc
from tokens import validated_request, AccessLevel
import tokens as token_manager

import uuid
import datetime
import json
from grpc_interceptor import ServerInterceptor
from grpc_reflection.v1alpha import reflection

import numerous_deployment_pb2, numerous_deployment_pb2_grpc
from kubernetes_api import kube_clusters

from numerous_cert_server.server import serve_env


def json_serial(obj):
    """JSON serializer for objects not serializable by default json code"""

    if isinstance(obj, (datetime.datetime, datetime.date)):
        return obj.timestamp()
    raise TypeError ("Type %s not serializable" % type(obj))

class InstanceInterceptor(ServerInterceptor):
    def intercept(self, method, request_or_iterator, context: grpc.ServicerContext, endpoint,  *args):

        context.metadata = {t[0]:t[1] for t in context.invocation_metadata()}
        log.debug('Requested endpoint: '+str(endpoint))
        return method(request_or_iterator, context)


log = logging.getLogger('numerous_kubernetes.server')
log.setLevel(os.getenv("NUMEROUS_LOGGING_LEVEL_SERVER") or logging.WARNING)

formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
streamhandler.setFormatter(formatter)
log.addHandler(streamhandler)


def standard_error_handling():
    def print_tb_(f):
        def ptb(self, request, context):
            try:
                tic = time.time()
                res = f(self, request, context)
                return res
            except:
                if not hasattr(self,'code') or self.code is None:
                    self.code = grpc.StatusCode.INTERNAL
                    self.msg = 'Internal error occurred in endpoint. Please check server logs.'
                    context.set_code(self.code)
                    context.set_details(self.msg)
                    raise
                else:
                    context.set_code(self.code)
                    context.set_details(self.msg)
                self.code = None
                self.msg = None
                return numerous_deployment_pb2.Empty()
            finally:
                toc = time.time()


        return ptb
    return print_tb_


class Error_Handler:
    def set_error(self, exception_cls, code: grpc.StatusCode=grpc.StatusCode.INTERNAL, msg:str='Internal Error Occured!'):
        self.code = code
        self.msg = msg

        raise exception_cls(self.msg)


class DeployServicer(numerous_deployment_pb2_grpc.DeployServicer, Error_Handler):

    @validated_request(access_level=AccessLevel.WRITE)
    @standard_error_handling()
    def CreateJob(self, request, context):
        spec = json.loads(request.json_spec)
        log.debug('Create from spec: '+str(spec))
        name = None
        namespace = None
        try:
            cluster = kube_clusters[spec['cluster']]
            spec.pop('cluster')
            name, namespace = cluster.create_job(**spec)
        except KeyError:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, "No such cluster! Available clusters are: "+str(list(kube_clusters.keys())))
        return numerous_deployment_pb2.CreateJobReply(json_reply=json.dumps({'status': 'OK', 'name': name, 'namespace': namespace}))

    @validated_request(access_level=AccessLevel.WRITE)
    @standard_error_handling()
    def SuspendJob(self, request, context):
        spec = json.loads(request.json_spec)

        try:
            cluster = kube_clusters[spec['cluster']]
            spec.pop('cluster')
            cluster.suspend_job(**spec)
        except KeyError:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, "No such cluster!")
        return numerous_deployment_pb2.SuspendJobReply(json_reply=json.dumps({'status': 'OK'}))

    @validated_request(access_level=AccessLevel.WRITE)
    @standard_error_handling()
    def DeleteJob(self, request, context):
        spec = json.loads(request.json_spec)

        try:
            cluster = kube_clusters[spec['cluster']]
            spec.pop('cluster')
            cluster.delete_job(**spec)
        except KeyError:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, "No such cluster!")
        return numerous_deployment_pb2.DeleteJobReply(json_reply=json.dumps({'status': 'OK'}))


    @validated_request(access_level=AccessLevel.WRITE)
    @standard_error_handling()
    def SetDeadlineJob(self, request, context):
        spec = json.loads(request.json_spec)

        try:
            cluster = kube_clusters[spec['cluster']]
            spec.pop('cluster')
            cluster.set_deadline_job(**spec)
        except KeyError:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, "No such cluster!")

        return numerous_deployment_pb2.SetDeadlineJobReply(json_reply=json.dumps({'status': 'OK'}))

    @validated_request(access_level=AccessLevel.READ)
    @standard_error_handling()
    def GetStatusJob(self, request, context):
        spec = json.loads(request.json_spec)

        cluster = None

        try:
            cluster = kube_clusters[spec['cluster']]
            spec.pop('cluster')

        except KeyError:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, "No such cluster!")

        return numerous_deployment_pb2.GetStatusJobReply(prefix=request.prefix, json_reply=json.dumps(cluster.get_job_status(**spec)))

    @validated_request(access_level=AccessLevel.READ)
    @standard_error_handling()
    def GetClusterInfo(self, request, context):

        return numerous_deployment_pb2.ClusterInfoReply(info=[
            numerous_deployment_pb2.ClusterInfo(name=key, os=cluster.os) for key, cluster in kube_clusters.items()
        ])

    @standard_error_handling()
    def GetAccessToken(self, request, context):
        access_token = None
        try:
            access_token = token_manager.generate_access_token(request.refresh_token.val)
        except token_manager.ValidationException as e:
            self.set_error(token_manager.ValidationException, grpc.StatusCode.UNAUTHENTICATED, e.__str__())
        except jwt.exceptions.DecodeError as e:
            self.set_error(token_manager.ValidationException, grpc.StatusCode.UNAUTHENTICATED, e.__str__())
        except KeyError as e:
            self.set_error(KeyError, grpc.StatusCode.NOT_FOUND, e.__str__())
        return numerous_deployment_pb2.Token(val=access_token)


def _initialize_channel(server, port):
    private_key = str.encode(os.getenv('NUMEROUS_CERT_KEY'))
    certificate_chain = str.encode(os.getenv('NUMEROUS_CERT_CRT'))
    serve_env('NUMEROUS_CERT_KEY', 'NUMEROUS_CERT_CRT', '0.0.0.0', 4443)
    server_credentials = grpc.ssl_server_credentials(((private_key, certificate_chain),))
    server.add_secure_port(f'[::]:{port}', server_credentials)
    reflection.enable_server_reflection(['DeployServicer'], server)
    return server


if __name__ == '__main__':
    # create a gRPC server
    server = grpc.server(futures.ThreadPoolExecutor(max_workers=100),interceptors=[InstanceInterceptor()])
    numerous_deployment_pb2_grpc.add_DeployServicer_to_server(DeployServicer(), server)
    server = _initialize_channel(server, 50051)
    server.start()

    log.info(f'Starting server. Listening on port {50051}')

    try:
        while True:
            time.sleep(86400)
    except KeyboardInterrupt:
        server.stop(0)

