import uuid
import grpc
from getpass import getpass

from numerous_api_client.python_protos.spm_pb2_grpc import TokenManagerStub
from numerous_api_client.python_protos.spm_pb2 import LoginCredentials, TokenRequest, RefreshRequest, Token

from .repository import NumerousRepository, ProjectScenario
from .utils import get_token_manager, yellow, bold


class AuthenticationError(Exception):
    pass


class TokenAuth(grpc.AuthMetadataPlugin):
    def __init__(self, access_token: str, instance_id: str = None):
        self._instance_id = instance_id or str(uuid.uuid4())
        self._access_token = access_token

    def __call__(self, _context: grpc.AuthMetadataContext, callback: grpc.AuthMetadataPluginCallback):
        callback((('token', self._access_token), ("instance", self._instance_id)), None)


def get_refresh_token(token_manager: TokenManagerStub, repo: NumerousRepository, email: str, password: str, scenario: ProjectScenario) -> str:
    token_request = TokenRequest(
        project_id = repo.scenario.project_id if scenario is None else scenario.project_id,
        scenario_id = repo.scenario.id if scenario is None else scenario.id,
        admin = False,
        execution_id = None,
        job_id = None,
        user_id = None,
        organization_id = repo.organization,
        agent = None,
        purpose = None,
        access_level = 3
    )

    credentials = LoginCredentials(
        email=email,
        password=password,
        token_request=token_request
    )
    
    try:
        return token_manager.LoginGetRefreshToken(credentials).val
    except grpc.RpcError:
        return None


def get_access_token(token_manager: TokenManagerStub, repo: NumerousRepository, refresh_token: str, scenario: ProjectScenario) -> str:
    refresh_request = RefreshRequest(
        refresh_token = Token(val=refresh_token),
        scenario_id = repo.scenario.id if scenario is None else scenario.id,
    )

    return token_manager.GetAccessToken(refresh_request).val


def login(repo: NumerousRepository, scenario: ProjectScenario = None):
    if repo.token is not None and (scenario is None or scenario == repo.scenario):
        try:
            with get_token_manager(repo.remote.api_url) as token_manager:
                return get_access_token(token_manager, repo, repo.token, scenario)
        except:
            pass

    print(yellow(f"Login to {bold(scenario or repo.scenario)} at {bold(repo.remote)}"))
    email = input("Email: ")
    password = getpass("Password: ")

    with get_token_manager(repo.remote.api_url) as token_manager:
        try:
            token = get_refresh_token(token_manager, repo, email, password, scenario)
            if token is None:
                raise AuthenticationError()
            repo.token = token
            repo.save()
            return get_access_token(token_manager, repo, repo.token, scenario)
        except grpc.RpcError as error:
            if error._state.code == grpc.StatusCode.UNAUTHENTICATED:
                raise AuthenticationError()
            raise