import io
import requests
from logging import Logger
from pathlib import Path

from numerous_api_client.python_protos.spm_pb2 import HistoryRequest, PullRequest

from ..repository import NumerousRepository, ProjectScenario, RepositoryRemote
from ..utils import *
from ..auth import AuthenticationError, TokenAuth, login


def command_clone(log: Logger, scenario: ProjectScenario, path: Path, remote: RepositoryRemote):
    path = path or Path.cwd()

    if path.exists():
        print(red(f'Cannot clone: {bold(path)} already exists'))
        return

    repo = NumerousRepository(path)
    repo.remote = remote

    try:
        access_token = login(repo)
    except AuthenticationError:
        print(red("Cannot clone: Login failed."))
        return

    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))
    with get_build_manager(path) as build_manager:
        history_request = HistoryRequest(repository=remote.name, scenario_id=scenario.id)
        updates = build_manager.GetHistory.with_call(history_request, credentials=call_credentials).updates

        if not updates:
            print(red('Cannot clone: No commit found'))
            return

        commit = updates[0]
        repo.snapshot = commit.snapshot_id
        repo.scenario = ProjectScenario(commit.scenario, commit.project)

        print(cyan(f'Downloading snapshot {bold(repo.snapshot)} into {bold(path)}'))
        pull_request = PullRequest(repository=repo.remote.name, snapshot_id=commit.snapshot_id)
        snapshot = build_manager.PullSnapshot.with_call(pull_request, credentials=call_credentials)
        response = requests.get(snapshot.archive_url.url, allow_redirects=True)
        extract_gzipped_tarball(io.BytesIO(response.content), path)

        repo.commit = commit.id
        repo.save()
        print(green(f"Repository {bold(repo.remote.name)} was cloned into {bold(path)}"))

