from logging import Logger
from datetime import datetime
from pathlib import Path

from prettytable import PrettyTable
from dirtools import filehash, Dir
from numerous_api_client.python_protos.spm_pb2 import HistoryRequest, HistoryUpdate

from ..repository import NumerousRepository
from ..utils import *
from ..auth import login, TokenAuth, AuthenticationError


def print_updates_table(repo: NumerousRepository, updates: list[HistoryUpdate]):
    dirty = Dir(str(repo.path)).hash(index_func=filehash) != repo.snapshot
    table = PrettyTable()
    table.field_names = ["Timestamp", "Scenario", "Comment", "Commit ID", "Current"]
    for update in updates:
        table.add_row([
            datetime.fromtimestamp(update.timestamp),
            update.scenario_id,
            update.comment,
            update.id,
            f"{green('*')} {red('dirty') if dirty else ''}" if repo.commit == update.id else ""])
    print(table)


def print_updates_git(repo: NumerousRepository, updates: list[HistoryUpdate]):
    dirty = Dir(str(repo.path)).hash(index_func=filehash) != repo.snapshot
    for update in updates:
        state = f" {green('current')}{red(' dirty') if dirty else ''}" if repo.commit == update.id else ""
        print(f"{yellow(f'commit {update.id}')} {yellow('(')}{cyan(update.project_id + ':' + update.scenario_id)}{yellow(')')}{state}")
        update_time = datetime.fromtimestamp(update.timestamp)
        print(f"Author: {update.user.full_name} <{update.user.email}>")
        print(f'Date:   {update_time.strftime("%a %b %d %H:%M:%S %Y %z")}')
        print()
        for line in update.comment.splitlines():
            print("    " + line)
        print()


def command_log(log: Logger, path: Path, display_mode: str):
    path = path or Path.cwd()

    try:
        repo = NumerousRepository(path)
        repo.load()
    except Exception as excp:
        log.debug(f"Exception {type(excp)} {excp}")
        print(red(f"Cannot get logs: {bold(str(path))} is not a repository."))
        return
    
    if repo.remote is None:
        print(red('Cannot get logs: No remote is configured for the repository.'))
        print(f"Use the command {bold('numbuild config --remote <REMOTE_URL>')} to configure a remote for the repository.")
        return

    if repo.scenario is None:
        print(red("Cannot get logs: A scenario must be checked out."))
        print(f"- Push to a scenario with {bold('numbuild push -c <COMMENT> <SCENARIO ID>')}")
        print(f"- Check out a scenario with {bold('numbuild checkout <SCENARIO ID>')}")
        return

    try:
        access_token = login(repo)
    except AuthenticationError:
        print(red("Cannot get logs: Login failed."))
        return
    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))
      
    with get_build_manager(repo.remote.api_url) as build_manager:
        history_request = HistoryRequest(repository=repo.remote.name, scenario_id=repo.scenario.id)
        history_reply, _call = build_manager.GetHistory.with_call(history_request, credentials=call_credentials)
    
    if not history_reply.updates:
        print(f"No history exists for remote {bold(repo.remote)}")
    
    if display_mode == 'table':
        print_updates_table(repo, history_reply.updates)
    else:
        print_updates_git(repo, history_reply.updates)
