import json
from logging import Logger, getLogger
from pathlib import Path
from datetime import datetime

import requests
from dirtools import DirState, filehash, Dir
from numerous_api_client.python_protos.spm_pb2 import Snapshot, SnapshotReply, HistoryUpdate

from ..repository import NumerousRepository
from ..utils import *
from ..auth import login, AuthenticationError, TokenAuth


log = getLogger("numerous.build-cli.push")


def get_snapshot_id(path: Path):
    dir = Dir(str(path))
    return dir.hash(index_func=filehash)


def get_snapshot_state(path: Path):
    dir = Dir(str(path))
    state = DirState(dir, index_cmp=filehash).state.copy()
    del state["directory"]
    return state


def command_push(log: Logger, path: Path, comment: str):
    path = Path.cwd() if path is None else path
    
    if not path.exists():
        print(red(f'Cannot push: {bold(path)} does not exist'))
        return

    repo = NumerousRepository(path)
    repo.load()
        
    if repo.scenario is None:
        print(red("Cannot push: A scenario must be checked out."))
        return

    if repo.snapshot == Dir(str(path)).hash(index_func=filehash):
        print(red(f'Cannot push: There is nothing new to push.'))
        return

    if repo.remote is None:
        print(red('Cannot push: No remote is configured for the repository.'))
        print(f"Use the command {bold('numbuild config --remote <REMOTE_URL>')} to configure a remote for the repository.")
        return

    try:
        access_token = login(repo)
    except AuthenticationError:
        print(red("Cannot push: Login failed."))
        return
    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))

    print(cyan(f"Pushing to remote repository {bold(repo.remote)}..."))
    print("Creating snapshot metadata...")
    snapshot_id = get_snapshot_id(repo.path)
    snapshot_state = get_snapshot_state(repo.path)
    snapshot = Snapshot(
        snapshot_id=snapshot_id,
        repository=repo.remote.name,
        file_structure=json.dumps(snapshot_state),
        only_get_uploads=False
    )

    with get_build_manager(repo.remote.api_url) as build_manager:
        try:
            snapshot_reply, _call = build_manager.CreateSnapshot.with_call(snapshot, credentials=call_credentials)
        except grpc.RpcError as error:
            if error.code() == grpc.StatusCode.ALREADY_EXISTS:
                log.info(f"Snapshot {snapshot_id} aleady exists. Ignoring.")
                print(yellow("Stopping. Snapshot already exists..."))
                return
            else:
                print(red("Cannot push: A connection error occured."))
                return

    print("Uploading snapshot...")
    with create_gzipped_tarball(repo.path) as tar_path:
        log.debug(f"Created tarball {tar_path} for {repo.path}")
        with open(tar_path, "rb") as tar_file:
            upload_response = requests.post(
                snapshot_reply.upload_url,
                data=tar_file,
                headers={"Content-Type": "multipart/related"},
                params={
                    "name": snapshot_id,
                    "mimeType": "application/octet-stream"
                },
            )
            if upload_response.ok:
                log.debug(f"Uploaded snapshot {snapshot_id} got response: {upload_response.status_code} {upload_response.reason}")
            else:
                log.debug(f"Failed uploading {snapshot_id} {upload_response.status_code} {upload_response.reason}")
                print(red(f"Cannot push: An error occured during upload"))
                return


    with get_build_manager(repo.remote.api_url) as build_manager:
        history_update = HistoryUpdate(
            repository=repo.remote.name,
            project_id=repo.scenario.project_id,
            scenario_id=repo.scenario.id,
            snapshot_id=snapshot_id,
            parent_commit_id=repo.commit,
            comment=comment,
            timestamp=datetime.utcnow().timestamp()
        )
        history_reply, _call = build_manager.CreateHistoryUpdate.with_call(history_update, credentials=call_credentials)
        log.debug(f'Got history reply {history_reply}')
    print(green(f'Pushed snapshot {bold(snapshot_id)} as commit {bold(history_reply.commit_id)}.'))

    repo.commit = history_reply.commit_id
    repo.snapshot = snapshot_id
    repo.save()
