import io
from logging import Logger
from pathlib import Path

from dirtools import filehash, Dir
from numerous_api_client.python_protos.spm_pb2 import HistoryRequest, PullRequest
import requests

from ..repository import NumerousRepository, ProjectScenario
from ..utils import *
from ..auth import login, TokenAuth, AuthenticationError


def command_checkout(log: Logger, path: Path, scenario: ProjectScenario, commit_id: str):
    path = Path.cwd() if path is None else path
    if not path.exists():
        print(red(f'Cannot checkout: {bold(path)} does not exist'))
        return
    
    repo = NumerousRepository(path)
    try:
        repo.load()
    except:
        print(red(f"Cannot checkout: {path} is not a repository."))
        return

    if repo.remote is None:
        print(red('Cannot checkout: No remote is configured for the repository.'))
        print(f"Use the command {bold('numbuild config --remote <REMOTE_URL>')} to configure a remote for the repository.")
        return

    try:
        access_token = login(repo, repo.scenario or scenario)
    except AuthenticationError:
        print(red("Cannot push: Login failed."))
        return
    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))

    with get_build_manager(repo.remote.api_url) as build_manager:
        history_request = HistoryRequest(
            repository = repo.remote.name,
            scenario_id = scenario.id or repo.scenario.id if commit_id is None else None,
            history_update_ids = None if commit_id is None else [commit_id]
        )
        
        history_reply, _call = build_manager.GetHistory.with_call(history_request, credentials=call_credentials)
        if not history_reply.updates:
            if scenario:
                repo.scenario = scenario
                repo.save()
                print(green(f"Checked out fresh scenario {bold(scenario)}"))
            else:
                print(red('Cannot checkout: no commit found'))
            return
        else:
            snapshot = Dir(str(path)).hash(index_func=filehash)
            if snapshot != repo.snapshot and input(yellow(f"There are changes since the last commit. Continue? (y/n) ")).strip().lower() != 'y':
                print(red('Aborted checkout'))
                return

        commit = history_reply.updates[0]

        print(cyan(f'Downloading snapshot {bold(commit.snapshot_id)} into {bold(path)}'))
        pull_request = PullRequest(repository=repo.remote.name, snapshot_id=commit.snapshot_id)
        snapshot, _call = build_manager.PullSnapshot.with_call(pull_request, credentials=call_credentials)
        response = requests.get(snapshot.archive_url.url, allow_redirects=True)
        extract_gzipped_tarball(io.BytesIO(response.content), path)

        repo.snapshot = commit.snapshot_id
        repo.commit = commit.id
        repo.scenario = commit.scenario
        repo.save()

        print(green(f"Checkout complete => Scenario {bold(repo.scenario)}, commit {bold(repo.commit)}"))
