import sys
import tarfile
from pathlib import Path, PurePosixPath
from io import BytesIO
from contextlib import contextmanager
from urllib.parse import urlparse
from uuid import uuid4
from os import linesep

from colorama import Style, Fore, init as init_colorama
from dirtools import Dir
import grpc

from numerous_api_client.python_protos import spm_pb2_grpc

from .repository import NumerousRepository


init_colorama()


def red(msg: str):
    return Fore.RED + msg + Fore.RESET


def green(msg: str):
    return Fore.GREEN + msg + Fore.RESET


def cyan(msg: str):
    return Fore.CYAN + msg + Fore.RESET


def yellow(msg: str):
    return Fore.YELLOW + msg + Fore.RESET


def bold(msg):
    return Style.BRIGHT + str(msg) + Style.NORMAL


def get_grpc_channel(url: str):
    parsed_url = urlparse(url)

    options = (
        ("grpc.keepalive_time_ms", 10000),
        ("grpc.keepalive_timeout_ms", 5000),
        ("grpc.keepalive_permit_without_calls", 1),
        ("grpc.http2_max_pings_without_data", 9999),
    )

    if parsed_url.scheme == "https":
        return grpc.secure_channel(
            f"{parsed_url.hostname}:{parsed_url.port or '443'}",
            grpc.ssl_channel_credentials(),
            options=options
        )
    else:
        return grpc.insecure_channel(
            f"{parsed_url.hostname}:{parsed_url.port or '80'}",
            options=options
        )


@contextmanager
def get_build_manager(api_url: str):
    channel = get_grpc_channel(api_url)
    try:
        yield spm_pb2_grpc.BuildManagerStub(channel)
    finally:
        channel.close()


@contextmanager
def get_token_manager(api_url: str):
    channel = get_grpc_channel(api_url)
    try:
        yield spm_pb2_grpc.TokenManagerStub(channel)
    finally:
        channel.close()


@contextmanager
def get_spm(api_url: str):
    channel = get_grpc_channel(api_url)
    try:
        yield spm_pb2_grpc.SPMStub(channel)
    finally:
        channel.close()


@contextmanager
def create_gzipped_tarball(source: Path) -> Path:
    source_dir = Dir(str(source))
    files = source_dir.files()
    tar_path = source / f"{uuid4()}.tar.gz"
    try:
        with tarfile.open(str(tar_path), mode="w:gz") as tar:
            for file_name in files:
                file_path = source / str(file_name)
                info = tarfile.TarInfo(str(PurePosixPath(Path(file_name))))
                info.size = file_path.stat().st_size
                with file_path.open("rb") as file:
                    tar.addfile(info, fileobj=file)
        yield tar_path
    finally:
        tar_path.unlink()


def display_error_and_exit(error_header: str, help_lines: list[str], exit_code: int = 1):
    print(red(error_header))
    print(*help_lines, sep=linesep)
    sys.exit(exit_code)


def extract_gzipped_tarball(file: BytesIO, path: Path) -> None:
    with tarfile.open(fileobj=file, mode="r:gz") as tar:
        tar.extractall(str(path))

