import json

from numerous_api_client.client import open_client
import os
from job_spec_formatter import format_job_spec, item_key_value

from time import time
import logging
import datetime
import traceback
from dotenv import load_dotenv
import random
load_dotenv()

log = logging.getLogger('main.simulation')
log.setLevel(logging.DEBUG)

def build_model_from_spec(job_id, scenario, model_definition, get_state, code_path):
    log.debug('Building model!')

    spec, run_settings = format_job_spec(job_id, scenario, model_definition)
    log.debug('spec ready')
    # Perform imports of code
    parameters = scenario['parameters']

    def get_val(param):
        return item_key_value(parameters, 'id', param)['value']

    # Read the stub (containing simulation data and model components) into a model objec

    old_state_applied = False
    state = get_state()
    if state is not None and 't' in state:
         log.info("Applying saved state!")
         old_state_applied = True
    else:
        log.info("No state applied.")

    return spec, state, old_state_applied, run_settings, spec['sim_specification']['sim_spec']['t_0']

class Debouncer:

    def __init__(self, debounce):
        self.debounce = debounce
        self.last = -999999

    def __call__(self, f, *args, **kwargs):
        tic = time()
        if tic > self.last+self.debounce:
            self.last = tic
            f(*args, **kwargs)

def test(scenario, model_definition, client, code_path, timeout):
    try:
        test_results = {'status': 'incomplete'}
        failed = False
        try:

            for job in scenario['jobs'].values():
                if job['isMain']:
                    test_results['execution'] = {'main_status': job['status']['status'], 'main_success': job['status']['status']=='finished'}
                    break
            else:
                test_results['execution'] = {'main_status': 'unknown', 'main_success': False}
                failed = True

            client.set_scenario_progress('Test report initializing', 'initializing', 0.0, force=True)


            spec, state, state_loaded, run_settings, t0 = build_model_from_spec(client._job_id, scenario, model_definition, client.get_state, code_path)


            log.debug('Retrieved spec and model.')

            sim_spec = spec['sim_specification']['sim_spec']

            df = client.data_read_df(tags=['_index'])

            t_end = sim_spec['t_end']

            def _status(a, b, prev_failed, fail_reason):
                failed = a != b
                return {'expected': a, 'actual': b, 'status': 'passed' if not failed else 'failed', 'reason': fail_reason if failed else 'passed'}, failed or prev_failed

            results = {}

            results['min'], failed = _status(t0, df['_index'].min(), failed, f'{t0} != {df["_index"].min()}')
            results['max'], failed = _status(t_end, df['_index'].max(), failed, f'{t_end} != {df["_index"].max()}')

            test_results['results'] = results

            test_results['status'] = 'failed' if failed else 'passed'
        finally:

            file_test_results = 'test_result.json'
            with open(file_test_results, 'w') as f:
                json.dump(test_results, f)

            client.upload_file(file_test_results, 'test_results', content_type='application/json')

        client.set_scenario_progress('Test failed' if failed else 'Test passed', 'finished' if not failed else 'failed', 100.0, force=True)

    except TimeoutError:
        client.set_scenario_progress('Test hibernated', 'finished', progress(), force=True)

    except (KeyboardInterrupt):
        client.set_scenario_progress('Test terminated', 'terminated', progress(), force=True)

    except Exception as e:

        tb = traceback.format_exc()
        print(tb)
        log.error(tb)
        client.set_scenario_progress('Test failed', 'failed', progress(), force=True)


def run(job_id=None, project=None, scenario=None, workpath="."):

    #Open the numerous client
    with open_client(job_id=job_id, project=project, scenario=scenario) as client:
        try:

            scenario, model_definition, files = client.get_scenario(path=workpath)
            os.chdir(workpath)



            test(scenario=scenario, model_definition=model_definition, client=client, code_path="", timeout=120)
            log.debug('Test completed')

        except (KeyboardInterrupt):
            client.set_scenario_progress('Test terminated', 'terminated', force=True)

        except Exception:
            tb = traceback.format_exc()
            log.error(tb)
            client.set_scenario_progress('Test failed', 'failed', force=True)

if __name__ == '__main__':
    run()