import json
from pathlib import Path, PurePosixPath
from typing import Optional

from urllib.parse import urlparse


class RepositoryRemote:

    def __init__(self, url: Optional[str] = None, api_url: Optional[str] = None,
                 job_id: Optional[str] = None, name: Optional[str] = None):
        if not any([val is None for val in [api_url, job_id, name]]):
            self.name = name
            self.api_url = name
            self.job_id = job_id
        elif url is not None:
            parsed_url = urlparse(url)
            _, self.job_id, self.name = PurePosixPath(parsed_url.path).parts
            self.api_url = f"{parsed_url.scheme}://{parsed_url.netloc}"
        else:
            raise ValueError("RepositoryRemote must be supplied 'url' or "
                             "'api_url', 'job_id' and 'name'.")
    
    def __str__(self):
        return f"{self.api_url}/{self.job_id}/{self.name}"


class ProjectScenario:

    def __init__(self, project_scenario: Optional[str] = None,
                 project_id: Optional[str] = None,
                 scenario_id: Optional[str] = None):
        if project_scenario:
            self.project_id, self.id = project_scenario.split(":")
        elif project_id is not None and scenario_id is not None:
            self.project_id = project_id
            self.id = scenario_id
        else:
            raise ValueError("ProjectScenario must be supplied "
                             "'project_scenario' or 'project_id' and "
                             "'scenario_id")

    def __str__(self):
        return f"{self.project_id}:{self.id}"


class NumerousRepository:
    FILE_NAME = ".numerous.repository.json"

    def __init__(self, path: Optional[Path] = None):
        self.path: Path = path or Path.cwd()
        self.organization: str = None
        self.user: str = None
        self.token: str = None
        self.remote: RepositoryRemote = None
        self.scenario: ProjectScenario = None
        self.commit: str = None
        self.snapshot: str = None

    def save(self):
        with (self.path / NumerousRepository.FILE_NAME).open("w") as fp:
            json.dump({
                "scenario": None if self.scenario is None else  str(self.scenario),
                "commit": self.commit,
                "snapshot": self.snapshot,
                "token": self.token,
                "remote": None if self.remote is None else str(self.remote),
                "user": self.user,
                "organization": self.organization,
                "user": self.user
            }, fp)
            return self

    def load(self):
        with (self.path / NumerousRepository.FILE_NAME).open("r") as fp:
            config = json.load(fp)
            self.remote = None if config["remote"] is None else \
                          RepositoryRemote(url=config["remote"])
            self.user = config["user"]
            self.organization = config["organization"]
            self.scenario = None if config["scenario"] is None else \
                            ProjectScenario(config["scenario"])
            self.commit = config["commit"]
            self.snapshot = config["snapshot"]
            self.token = config["token"]
            return self

    def get_client(self, clear_data: Optional[bool] = None,
                    no_log: bool = False, instance_id: Optional[str] = None):
        """
        Returns a `NumerousClient` instance configured to connect to the API
        at the remote of this repository. All arguments are passed on to the
        client constructor.
        """
        from numerous_api_client.client import NumerousClient
        return NumerousClient(url=self.remote.api_url,
                              project=self.scenario.project_id,
                              scenario=self.scenario.id,
                              job_id=self.remote.job_id,
                              refresh_token=self.token, clear_data=clear_data,
                              no_log=no_log, instance_id=instance_id)
    
    def get_admin_client(self):
        """
        Returns a `NumerousAdminClient` instance configured to connect to the
        API at the remote of this repository.
        """
        from numerous_api_client.client import NumerousAdminClient
        admin_client = NumerousAdminClient(url=self.remote.api_url,
                                           refresh_token=self.token)
        return admin_client
