import json
from logging import Logger
from pathlib import Path

from numerous_api_client.python_protos.spm_pb2 import BuildRequest, BuildInfoRequest, Subscription, JobImageReferenceUpdate

from ..repository import NumerousRepository
from ..utils import *
from ..auth import login, AuthenticationError, TokenAuth


class BuildError(Exception):
    pass


def handle_build_log_message(msg):
    msg = json.load(msg)
    if msg == ":BUILD_DONE:":
        return True
    elif msg == ":ERROR:":
        raise BuildError()
    else:
        print(cyan("[Remote builder]"), msg)
    return False


def command_build(log: Logger, path: Path, commit: str, job: str = None, launch: bool = False):
    path = Path.cwd() if path is None else path
    repo = NumerousRepository(path)
    try:
        repo.load()
    except:
        print(red(f'Cannot build: {bold(path)} does not exist.'))
        return

    if repo.commit is None and commit is None:
        print(red("Cannot build: No commit checked out or specified."))
        return   

    if repo.remote is None:
        print(red('Cannot build: No remote is configured for the repository.'))
        print(f"Use the command {bold('numbuild config --remote <REMOTE_URL>')} to configure a remote for the repository.")
        return

    if repo.snapshot is None or repo.commit is None:
        print(red("Cannot build: No push has been made to the remote."))
        return

    try:
        access_token = login(repo)
    except AuthenticationError:
        print(red("Cannot build: Login failed."))
        return
    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))
    
    with get_build_manager(repo.remote.api_url) as build_manager:
        build_request = BuildRequest(
            repository=repo.remote.name,
            commit_id=commit or repo.commit,
            launch_after_build=launch
        )
        build_reply, _call = build_manager.LaunchBuild.with_call(build_request, credentials=call_credentials)
        build_id = build_reply.build_id

        print(cyan(f"Build {bold(build_id)} enqueued. Awaiting remote builder..."))
        with get_spm(repo.remote.api_url) as spm:
            subscription = Subscription(channel_patterns=[build_id], project_id=repo.scenario.project_id, scenario=repo.scenario.id)
            for message in spm.SubscribeForUpdates(subscription, credentials=call_credentials):
                try:
                    log.debug("Got build message: %s", message)
                    build_done = handle_build_log_message(message.message)
                    if build_done:
                        break
                except BuildError:
                    print(red("Build failed: An error occured during the build."))
                    return

        build_info_request = BuildInfoRequest(repository=repo.remote.name, build_id=build_id)
        build_info_response, _call = build_manager.GetBuildInfo.with_call(build_info_request, credentials=call_credentials)
        build_info = json.loads(build_info_response.info)

        print(cyan("Updating job image reference..."))
        job_image_reference_update = JobImageReferenceUpdate(
            project = repo.scenario.project_id,
            scenario = repo.scenario.id,
            job = repo.remote.job_id,
            name = repo.remote.name,
            path = build_info['tags'][0],
            repository = repo.remote.name,
            commit_id = repo.commit,
            internal = True,
            build = build_id
        )
        build_manager.UpdateJobImageReference.with_call(job_image_reference_update, credentials=call_credentials)
    print(green(f"Success: Build {bold(build_id)} complete"))
