import uuid
from getpass import getpass
from typing import Optional

import grpc
from numerous_api_client.python_protos.spm_pb2 import LoginCredentials, RefreshRequest, Token, TokenRequest
from numerous_api_client.python_protos.spm_pb2_grpc import TokenManagerStub

from .repository import NumerousRepository, ProjectScenario
from .utils import bold, get_token_manager, yellow


class AuthenticationError(Exception):
    pass


class TokenAuth(grpc.AuthMetadataPlugin):
    def __init__(self, access_token: str, instance_id: str = None):
        self._instance_id = instance_id or str(uuid.uuid4())
        self._access_token = access_token

    def __call__(self, _context: grpc.AuthMetadataContext, callback: grpc.AuthMetadataPluginCallback):
        callback((('token', self._access_token), ("instance", self._instance_id)), None)


def get_refresh_token(token_manager: TokenManagerStub, repo: NumerousRepository, email: str, password: str,
                      scenario: Optional[ProjectScenario] = None) -> Optional[str]:

    if scenario is None:
        if repo.scenario is None:
            return None
        else:
            scenario_id = repo.scenario.id
            project_id = repo.scenario.project_id
    else:
        scenario_id = scenario.id
        project_id = scenario.project_id

    token_request = TokenRequest(
        project_id=project_id,
        scenario_id=scenario_id,
        execution_id=f"numerous_cli-{uuid.uuid4()}",
        admin=False,
        job_id=repo.remote.job_id if repo.remote is not None else None,  # TODO: maybe fail early if remote not defined
        user_id=None,
        organization_id=repo.organization,
        agent=None,
        purpose=None,
        access_level=3
    )

    credentials = LoginCredentials(
        email=email,
        password=password,
        token_request=token_request
    )

    try:
        return str(token_manager.LoginGetRefreshToken(credentials).val)
    except grpc.RpcError:
        return None


def get_access_token(token_manager: TokenManagerStub, repo: NumerousRepository, refresh_token: str,
                     scenario: Optional[ProjectScenario] = None) -> str:
    scenario = scenario or repo.scenario
    if scenario is None:
        raise AuthenticationError()
    refresh_request = RefreshRequest(refresh_token=Token(val=refresh_token), scenario_id=scenario.id)
    return str(token_manager.GetAccessToken(refresh_request).val)


def login(repo: NumerousRepository, scenario: Optional[ProjectScenario] = None):
    if repo.remote is None:
        raise AuthenticationError()

    scenario = scenario or repo.scenario
    if scenario is None:
        raise AuthenticationError()

    if repo.token is not None:
        try:
            with get_token_manager(repo.remote.api_url) as token_manager:
                return get_access_token(token_manager, repo, repo.token, scenario)
        except Exception:  # nosec
            pass

    print(yellow(f"Login to {bold(scenario)} at {bold(repo.remote)}"))
    email = input("Email: ")  # nosec
    password = getpass("Password: ")

    with get_token_manager(repo.remote.api_url) as token_manager:
        try:
            token = get_refresh_token(token_manager, repo, email, password, scenario)
            if token is None:
                raise AuthenticationError()
            repo.token = token
            repo.save()
            return get_access_token(token_manager, repo, repo.token, scenario)
        except grpc.RpcError as error:
            if error._state.code == grpc.StatusCode.UNAUTHENTICATED:
                raise AuthenticationError()
            raise
