import io
from logging import Logger
from pathlib import Path

import grpc
import requests
from dirtools import Dir, filehash
from numerous_api_client.python_protos.spm_pb2 import HistoryRequest, PullRequest

from ..auth import AuthenticationError, TokenAuth, login
from ..repository import NumerousRepository, ProjectScenario
from ..utils import bold, cyan, extract_gzipped_tarball, get_build_manager, green, red, yellow


def command_checkout(log: Logger, path: Path, scenario: ProjectScenario, commit_id: str):
    path = Path.cwd() if path is None else path
    if not path.exists():
        print(red(f'Cannot checkout: {bold(path)} does not exist'))
        return

    repo = NumerousRepository(path)
    try:
        repo.load()
    except Exception:
        print(red(f"Cannot checkout: {path} is not a repository."))
        return

    if repo.remote is None:
        print(red('Cannot checkout: No remote is configured for the repository.'))
        print(f"Use the command {bold('numbuild config --remote <REMOTE_URL>')} to configure a remote for the "
              "repository.")
        return

    try:
        access_token = login(repo, repo.scenario or scenario)
    except AuthenticationError:
        print(red("Cannot push: Login failed."))
        return
    call_credentials = grpc.metadata_call_credentials(TokenAuth(access_token))

    with get_build_manager(repo.remote.api_url) as build_manager:
        scenario_id = None
        if commit_id is None:
            if scenario is not None:
                scenario_id = scenario.id
            elif repo.scenario is not None:
                scenario_id = repo.scenario.id
        history_request = HistoryRequest(
            repository=repo.remote.name,
            scenario_id=scenario_id,
            history_update_ids=None if commit_id is None else [commit_id]
        )

        history_reply, _call = build_manager.GetHistory.with_call(history_request, credentials=call_credentials)
        if not history_reply.updates:
            if scenario:
                repo.scenario = scenario
                repo.save()
                print(green(f"Checked out fresh scenario {bold(scenario)}"))
            else:
                print(red('Cannot checkout: no commit found'))
            return
        else:
            snapshot = Dir(str(path)).hash(index_func=filehash)
            choice = input(yellow("There are changes since the last commit. Continue? (y/n) "))  # nosec
            if snapshot != repo.snapshot and choice.strip().lower() != 'y':
                print(red('Aborted checkout'))
                return

        commit = history_reply.updates[0]

        print(cyan(f'Downloading snapshot {bold(commit.snapshot_id)} into {bold(path)}'))
        pull_request = PullRequest(repository=repo.remote.name, snapshot_id=commit.snapshot_id)
        snapshot, _call = build_manager.PullSnapshot.with_call(pull_request, credentials=call_credentials)
        response = requests.get(snapshot.archive_url.url, allow_redirects=True)
        extract_gzipped_tarball(io.BytesIO(response.content), path)

        repo.snapshot = commit.snapshot_id
        repo.commit = commit.id
        repo.scenario = ProjectScenario(scenario_id=commit.scenario_id, project_id=commit.project_id)
        repo.save()

        print(green(f"Checkout complete => Scenario {bold(repo.scenario)}, commit {bold(repo.commit)}"))
