from sim_tools.assemble_solve.model_assemble import Model
from sim_tools.store import  simulation_store_mongodb
from sim_components.generic.items import SubsystemPrescribed
import jsonpickle
from numerous_engine_0.dependencies.job_spec_formatter import format_job_spec, item_key_value
import importlib
from time import time
import logging
import datetime
import traceback

log = logging.getLogger('numerous0.simulation')
log.setLevel(logging.DEBUG)

def build_model_from_spec(job_id, scenario, model_definition, get_state, code_path) -> Model:
    log.debug('Building model!')




    spec, run_settings = format_job_spec(job_id, scenario, model_definition)
    log.debug('spec ready')
    # Perform imports of code
    parameters = scenario['parameters']

    def get_val(param):
        return item_key_value(parameters, 'id', param)['value']

    #for k, v in get_val('imports').items():
    imports_made = []
    for s in scenario['simComponents']:
        module_ = s['item_class'].split('.')[:-1]

        if not (imp:= ".".join(module_)) in imports_made:
            imports_made.append(imp)

            location = code_path+"/" + "/".join(module_)+'.py'
            file = module_[-1]

            spec_ = importlib.util.spec_from_file_location(file, location)

            globals()[file] = importlib.util.module_from_spec(spec_)
            spec_.loader.exec_module(globals()[file])



    item_fact = simulation_store_mongodb.Item_Factory('')

    recipe = jsonpickle.loads(spec['sim_specification']['model_spec']['kwargs'])
    print('recipe: ', recipe)
    stub = item_fact.get_Item('Stub', '', SubsystemPrescribed, **recipe)

    # Read the stub (containing simulation data and model components) into a model object
    model = Model('model', stub)
    model.assemble({'no_tmp': True, 'level': 3})

    model.apply_init_state({'t': spec['sim_specification']['sim_spec']['t_0']})

    old_state_applied = False
    state = get_state()
    if state is not None and 'system_state' in state:
         log.info("Applying saved state!")
         model.apply_init_state(state['system_state'])
         old_state_applied = True
    else:
        log.info("No state applied.")

    model.set_stop_tag(spec['sim_specification']['sim_spec']['stop_tag'])

    return spec, model, old_state_applied, run_settings

class Debouncer:

    def __init__(self, debounce):
        self.debounce = debounce
        self.last = -999999

    def __call__(self, f, *args, **kwargs):
        tic = time()
        if tic > self.last+self.debounce:
            self.last = tic
            f(*args, **kwargs)

def simulate_model(scenario, model_definition, output, client, code_path, timeout):
    client.set_scenario_progress('Simulation initializing', 'initializing', 0.0, force=True)


    spec, model, state_applied, run_settings = build_model_from_spec(client._job_id, scenario, model_definition, client.get_state, code_path)
    log.debug('Retrieved spec and model.')

    sim_spec = spec['sim_specification']['sim_spec']

    log.debug('simspce: ' + str(sim_spec))

    dt = sim_spec['dt']
    t_end = sim_spec['t_end']
    t_0 = model.t
    log.debug('t0: '+ str(model.t))
    log.debug('t end : ' + str(t_end))

    model.next_out = model.t + dt

    dt_int = sim_spec['dt_internal']

    offset = datetime.datetime.fromisoformat(run_settings['startDate']).timestamp()

    def apply_row(row):
        last_t = row['_index']
        row.pop('_index')
        row.pop('_index_offset')
        row.pop('_datetime_utc')

        model.apply_init_state(row)
        return last_t

    first_call = {'is_first': True}

    def output_(data):
        data['_index']=data['t']# + 1609459200
        output.write_row(data)

        if first_call['is_first']:
            first_call['is_first'] = False
            #client.set_scenario_data_tags(data.keys())
            client.set_timeseries_meta_data([{'name': t} for t in data.keys()], {}, offset=offset, timezone='UTC', epoch_type='s')

    def set_state_(force=False):
        state = model.get_combined_dict()
        client.set_state({'system_state': state}, force=force)

    def progress():
        return round((model.t- t_0) / (t_end - t_0) * 100, 0)

    client.set_scenario_progress('Simulation in progress', 'running', progress(), force=True)

    debounce = Debouncer(300)
    debounce(set_state_, True)

    log.debug('Starting simulation')

    try:
        last_t =  None
        log.debug('Starting at t: '+str(model.t))
        for row in client.iter_inputs(scenario, t0=model.t+offset, te=t_end+offset, dt=dt, tag_prefix='Stub', tag_seperator='.', timeout=timeout, subscribe=True):
            #log.debug('Consuming _index: '+str(row['_index']))
            #row['_index']*=3600
            if last_t is not None:
                dt_solve = row['_index'] - last_t
                model.step_update_output(dt_solve, dt, float(dt_int), history=False, output = output_)

                client.set_scenario_progress('Simulation in progress', 'running', progress())

                debounce(set_state_)


                if model.t >= t_end:
                    log.debug(f't end of {t_end} s reached')
                    log.debug('Time spent on simulation: '+ str(model.step_time))
                    log.debug('Time spent on overhead: '+ str(model.non_step_time))
                    break


            last_t = apply_row(row)

        log.debug(f'Simulation finished at t: {model.t} s')
        client.set_scenario_progress('Simulation completed', 'finished', 100.0, force=True)
        #client.upload_file('./Alingsas_Alingsas_102204_2019_Sveby.csv', 'Alingsas_Alingsas_102204_2019_Sveby.csv')

    except TimeoutError:
        client.set_scenario_progress('Simulation hibernated', 'finished', progress(), force=True)

    except (KeyboardInterrupt):
        client.set_scenario_progress('Simulation terminated', 'terminated', progress(), force=True)

    except Exception as e:
        tb = traceback.format_exc()
        log.error(tb)
        client.set_scenario_progress('Simulation failed', 'failed', progress(), force=True)


    finally:
        set_state_(force=True)
