from dependency_injector import containers, providers
from dataclasses import dataclass
from sim_tools.store.simulation_store_mongodb import Item_Factory
from enum import Enum

def fullname(o):
  # o.__module__ + "." + o.__class__.__qualname__ is an example in
  # this context of H.L. Mencken's "neat, plausible, and wrong."
  # Python makes no guarantees as to whether the __module__ special
  # attribute is defined, so we take a more circumspect approach.
  # Alas, the module name is explicitly excluded from __qualname__
  # in Python 3.

  module = o.__class__.__module__
  if module is None or module == str.__class__.__module__:
    return o.__class__.__name__  # Avoid reporting __builtin__
  else:
    return module + '.' + o.__class__.__name__

class ModelContainer(containers.DynamicContainer):

    def __init__(self):
        super().__init__()

        self.meta = {}

    def add_model(self, name, meta, func):
        if not name:
            name = func.__module__ + '.' + func.__name__

        if not name in self.meta:
            setattr(self, name, providers.Factory(func))
        else:
            raise KeyError(f'Duplicate model name {name} found! Model needs unique name to be registered')

        self.meta[name] = meta


model_container = ModelContainer()


class ParamType(Enum):
    INT = 0
    FLOAT = 1
    STRING = 2
    FILE = 3
    REFERENCE = 4
    BOOLEAN = 5
    SELECTOR = 6
    CONFIG = 7
    MONTH = 8

    def to_string(self):
        return self.name.lower() if self.name not in ['INT', 'FLOAT'] else 'number'


class InputType(Enum):
    DYNAMIC = 0
    STATIC = 1

    def to_string(self):
        return self.name.lower()


class Month(Enum):
    JANUARY = 1
    FEBRUARY = 2
    MARCH = 3
    APRIL = 4
    MAY = 5
    JUNE = 6
    JULY = 7
    AUGUST = 8
    SEPTEMBER = 9
    OCTOBER = 10
    NOVEMBER = 11
    DECEMBER = 12

    def to_string(self):
        return self.name.lower().capitalize()

@dataclass
class Parameter:
    id: str
    display_name: str = ""
    tooltip: str = ""
    param_type: ParamType = ParamType.FLOAT
    default_value: object = 0
    optional: bool = False

    def to_dict(self):

        # Allow for data type "Month" to be used an properly represented
        if isinstance(self.default_value, Month):
            self.default_value = self.default_value.to_string()

        return {
            'displayName': self.display_name, 'id': self.id, 'tooltip': self.tooltip,
            'type': self.param_type.to_string(), 'value': self.default_value
        }


@dataclass
class Input_Variable:
    id: str
    offset: float = 0
    scaling: float = 1
    source_type: InputType = InputType.DYNAMIC
    tooltip: str = ""
    unit: str = ""
    display_name: str = ""
    default_value: float = 0.0
    optional: bool = False

    def to_dict(self):
        return {
            'display': self.display_name, 'id': self.id, 'offset': self.offset,
            'scaling': self.scaling, 'sourceType': self.source_type.to_string(),
            'tooltip': self.tooltip, 'unit': self.unit, 'value': self.default_value
        }

@dataclass
class ModelMeta:
    type: str
    display_name: str
    class_name: str
    module: str
    parameters: tuple = tuple()
    input_variables: tuple = tuple()

    def to_dict(self):

        # Make sure input_variables and parameters are tuples (or none if empty)
        if self.parameters is not None and not isinstance(self.parameters, tuple):
            self.parameters = (self.parameters, )

        if self.input_variables is not None and not isinstance(self.input_variables, tuple):
            self.input_variables = (self.input_variables, )

        # Return a dict version of object
        return {
            'name': self.type, 'type': self.type,
            'item_class': self.module.split('.')[-1] + '.' + self.class_name,
            'displayName': self.display_name,
            'parameters': [param.to_dict() for param in self.parameters] if self.parameters is not None else [],
            'inputVariables': [inp.to_dict() for inp in self.input_variables] if self.input_variables is not None else []
        }


#Model decorator
def model(type=None, name=None, display_name=None, parameters= None, input_variables = None):
    def inner(func):

        # The line '[display_name, func.__name__][display_name is None]' sets the display_name
        # to a default of the function name, if the display_name is not set (None)
        model_container.add_model(
            name,
            ModelMeta(
                type,
                [display_name, func.__name__][display_name is None],
                func.__name__, func.__module__, parameters, input_variables
            ),
            func
        )
        return func

    return inner



