import logging
import time

import grpc
from google.protobuf.json_format import MessageToDict
from grpc_interceptor import ServerInterceptor


class Interceptor(ServerInterceptor):
    def __init__(self, logger: logging.Logger):
        self.logger = logger

    def intercept(self, method, request_or_iterator, context: grpc.ServicerContext, endpoint, *args):
        context.metadata = {t[0]: t[1] for t in context.invocation_metadata()}
        self.logger.debug('Requested endpoint: %s', endpoint)
        try:
            msg = MessageToDict(request_or_iterator)
            if 'data' in msg:
                msg.pop('data')
                msg.update({'note': "data omitted"})
            self.logger.debug(f'endpoint {endpoint} - request: {msg}')
        except Exception as exception:
            self.logger.error(f'Could not translate message {exception}')
        return method(request_or_iterator, context)

    def intercept_service(self, continuation, handler_call_details):
        start = time.time()
        self.logger.info("Request %s", handler_call_details.method)
        self.logger.debug("Invocation-Metadata for %s: %s", handler_call_details.method,
                          ", ".join([f'{m.key}={m.value}' for m in handler_call_details.invocation_metadata]))
        try:
            return continuation(handler_call_details)
        finally:
            self.logger.debug('Executed %s, duration %.3fms', handler_call_details.method, 1000 * (time.time() - start))
