import datetime
import logging
from functools import wraps
from pprint import PrettyPrinter
from typing import Any, Dict, Iterable, Optional, Type

import grpc
from google.protobuf.json_format import MessageToDict

from numerous.tokens import BaseValidationError

from numerous.server_common.exceptions import AlreadyExistsError, InvalidArgumentError, NotFoundError, \
    NumerousBaseError, UnknownError

log = logging.getLogger(__name__)


def json_serial(obj):
    """JSON serializer for objects not serializable by default json code"""
    if isinstance(obj, (datetime.datetime, datetime.date)):
        return obj.timestamp()
    raise TypeError("Type %s not serializable" % type(obj))


BASE_ERROR_TO_GRPC_STATUS_CODE: dict[Type[Exception], grpc.StatusCode] = {
    AlreadyExistsError: grpc.StatusCode.ALREADY_EXISTS,
    InvalidArgumentError: grpc.StatusCode.INVALID_ARGUMENT,
    NotFoundError: grpc.StatusCode.NOT_FOUND,
    UnknownError: grpc.StatusCode.UNKNOWN,
    BaseValidationError: grpc.StatusCode.UNAUTHENTICATED
}


def error_to_grpc_status_code(e: Exception) -> grpc.StatusCode:
    for error_type, code in BASE_ERROR_TO_GRPC_STATUS_CODE.items():
        if isinstance(e, error_type):
            return code
    return grpc.StatusCode.UNKNOWN


def getattr_short_circuit(o: Any, attrs: Iterable[str]) -> Any:
    """Get a named attribute from an object request (takes a list of values and tries to get them in order)."""
    for attr in attrs:
        try:
            return getattr(o, attr)
        except AttributeError:
            pass


def handle_errors(func):
    @wraps(func)
    def handle_errors(self, request: Any, context: grpc.ServicerContext):
        try:
            return func(self, request, context)
        except (NumerousBaseError, BaseValidationError) as error:
            details = get_details_with_hints(error.msg, request, self)
            code = error_to_grpc_status_code(error)
        except Exception as e:
            try:
                raw_request_dump: Optional[Dict[str, Any]] = MessageToDict(request)
            except AttributeError:
                raw_request_dump = MessageToDict(request[0]) if request else None
            request_dump = PrettyPrinter(indent=2, width=120, depth=6).pformat(raw_request_dump)
            log.exception("Unhandled %s occurred handling request: %s", type(e).__name__, request_dump)
            details = get_details_with_hints('Internal error occurred in endpoint. Please check server logs.', request,
                                             self)
            code = grpc.StatusCode.INTERNAL
        log.info("Aborting context. Code: %s, details: %s", code, details)
        context.abort(code, details)

    return handle_errors


def get_details_with_hints(msg: str, request, servicer) -> str:
    firebase = getattr(servicer, "firebase_service", None)
    hints = []
    if not hasattr(request, 'project_id'):
        log.warning("Request had no project_id attribute")
    elif firebase and request.project_id and not firebase.check_project_exist(request.project_id):
        log.warning(f"Request targeted non-existent project {request.project_id}")
        hints.append(f'Project {request.project_id} does not exist!')
    elif not hasattr(request, 'scenario_id'):
        log.warning("Request had no scenario_id attribute")
    elif firebase and not firebase.check_scenario_exist(request.project_id, request.scenario_id):
        log.warning(f"Request targeted non-existent scenario {request.scenario_id}")
        hints.append(f'Scenario {request.scenario_id} does not exist!')
    elif not hasattr(request, 'job_id'):
        log.warning("Request had no job_id attribute")
    elif firebase and not firebase.check_job_exist(request.project_id, request.scenario_id, request.job_id):
        log.warning(f"Request targeted non-existent job {request.job_id}")
        hints.append(f'Job {request.job_id} does not exist!')

    hints = ["Hints:"] + hints if hints else []
    return "\n".join([msg] + hints)
