import abc
import logging
from functools import wraps
from typing import Any, Callable, Iterable, Optional, Tuple, TypeVar

from numerous.tokens.exceptions import ValidationFailedError
from numerous.tokens.token_manager import TokenManager
from numerous.tokens.token_validation import AccessLevel, UserRole, ValidationObjectType

from .utilities import getattr_short_circuit

log = logging.getLogger(__name__)


def _get_tokens_from_metadata(metadata: Iterable[Any]) -> Tuple[Optional[str], Optional[str]]:
    access_token = None
    id_token = None
    for meta in metadata:
        if meta.key == 'token':
            access_token = meta.value
        elif meta.key == 'authorization':
            id_token = meta.value
    return access_token, id_token


TAuthorizer = TypeVar('TAuthorizer', bound='Authorizer')


class Authorizer(abc.ABC):

    @abc.abstractmethod
    def authorize(self, request: Any, context: Any, access_level: AccessLevel, user_role: Optional[UserRole],
                  validation_object_type: ValidationObjectType):
        pass


class TokenAuthorizer(Authorizer):

    def __init__(self, token_manager: TokenManager):
        self._token_manager: TokenManager = token_manager

    @property
    def token_manager(self) -> TokenManager:
        return self._token_manager

    def authorize(self, request: Any, context: Any, access_level: AccessLevel, user_role: Optional[UserRole],
                  validation_object_type: ValidationObjectType):
        project_id = getattr_short_circuit(request, ['project_id', 'project'])
        scenario_id = getattr_short_circuit(request, ['scenario_id', 'scenario'])
        prefix = getattr(request, 'prefix', None)
        access_token, id_token = _get_tokens_from_metadata(context.invocation_metadata())
        claims = self.token_manager.authorize_get_claims(project_id, scenario_id, prefix, access_token,
                                                         id_token, access_level, user_role,
                                                         validation_object_type)
        context.claims = claims


def authorize_grpc_request(
        access_level: AccessLevel,
        user_role: Optional[UserRole] = None,
        validation_object_type: ValidationObjectType = ValidationObjectType.SCENARIO
) -> Callable[[Callable[[TAuthorizer, Any, Any], Any]], Callable[[TAuthorizer, Any, Any], Any]]:
    """Decorator that can be added to an endpoint to enable customizable validation"""

    def wrapper(method):
        @wraps(method)
        def wrapped(servicer: TAuthorizer, request, context):
            try:
                servicer.authorize(request, context, access_level, user_role, validation_object_type)
            except ValidationFailedError as e:
                log.info('Could not authorize token for request %s, AccessLevel=%s, UserRole=%s, '
                         'ValidationObjectType=%s - %s', method.__qualname__, access_level.name,
                         user_role.name if user_role else None, validation_object_type.name, e.msg)
                raise
            return method(servicer, request, context)

        return wrapped

    return wrapper


def authorize_grpc_stream(
        access_level: AccessLevel,
        user_role: Optional[UserRole] = None,
        validation_object_type: ValidationObjectType = ValidationObjectType.SCENARIO,
) -> Callable:
    def wrapper(method):
        @wraps(method)
        def wrapped(servicer: Authorizer, request_iterator, context):
            def authorize_grpc_request_iterator():
                for request in request_iterator:
                    try:
                        servicer.authorize(request, context, access_level, user_role, validation_object_type)
                    except ValidationFailedError as e:
                        user_role_name = user_role.name if user_role else None
                        log.info('Could not authorize token for stream %s, AccessLevel=%s, UserRole=%s, '
                                 'ValidationObjectType=%s - %s', method.__qualname__, access_level.name, user_role_name,
                                 validation_object_type.name, user_role_name, e.msg)
                        raise
                    yield request

            return method(servicer, authorize_grpc_request_iterator(), context)

        return wrapped

    return wrapper
