import datetime
from enum import Enum, IntEnum

import jwt

from .exceptions import DecodeTokenError, ExpiredTokenError

REFRESH_TOKEN_EXP = None  # datetime.timedelta(hours=24)
ACCESS_TOKEN_EXP = datetime.timedelta(minutes=10)
TOKEN_ALGORITHM = "HS256"  # nosec


class TokenType(str, Enum):
    REFRESH = "REFRESH"
    ACCESS = "ACCESS"


class AccessLevel(IntEnum):
    ANY = 0
    READ = 1
    WRITE = 2
    DEVELOPER = 3
    OWNER = 4
    ADMIN = 5


class UserRole(str, Enum):
    GUEST = 'guest'
    SIMULATOR = 'simulator'
    DEVELOPER = 'developer'
    OWNER = 'owner'
    PORTFOLIO_USER = 'portfolio_user'
    PORTFOLIO_MANAGER = 'portfolio_manager'


def encode_token(payload: dict, key: str, algorithm: str = TOKEN_ALGORITHM) -> str:
    return jwt.encode(payload, key=key, algorithm=algorithm)  # type: ignore[return-value]


def decode_token(token: str, key: str, algorithm: str = TOKEN_ALGORITHM) -> dict:
    try:
        return jwt.decode(jwt=token, key=key, algorithms=[algorithm])
    except jwt.DecodeError as e:
        raise DecodeTokenError(e)
    except jwt.ExpiredSignatureError:
        raise ExpiredTokenError()


def decode_token_no_verify(token: str) -> dict:
    return jwt.decode(token, algorithms=[TOKEN_ALGORITHM], options={'verify_signature': False})
