import abc
import datetime
import logging
from typing import Optional, Tuple

from numerous.tokens import token_validation
from numerous.tokens.common import ACCESS_TOKEN_EXP, AccessLevel, TokenType, UserRole, decode_token, encode_token
from numerous.tokens.exceptions import MissingTokenError, ValidationFailedError
from numerous.tokens.models import ApiClaims
from numerous.tokens.token_validation import ValidationObjectType

log = logging.getLogger(__name__)

USER_ROLE_TO_ACCESS_LEVEL = {
    UserRole.GUEST: AccessLevel.READ,
    UserRole.PORTFOLIO_USER: AccessLevel.READ,
    UserRole.PORTFOLIO_MANAGER: AccessLevel.WRITE,
    UserRole.DEVELOPER: AccessLevel.DEVELOPER,
    UserRole.OWNER: AccessLevel.ADMIN,
    UserRole.SIMULATOR: AccessLevel.WRITE,
}


class IdTokenVerifier(abc.ABC):

    @abc.abstractmethod
    def verify_get_uid(self, id_token: str) -> Optional[str]:
        pass


class TokenManager:

    def __init__(self, secret: str, id_token_verifier: Optional[IdTokenVerifier],
                 user_access_provider: Optional[token_validation.UserAccessProvider]):
        self._secret = secret
        self._id_token_verifier = id_token_verifier
        self._user_access_provider = user_access_provider

    def authorize_get_claims(
            self,
            project_id: Optional[str],
            scenario_id: Optional[str],
            prefix: Optional[str],
            access_token: Optional[str],
            id_token: Optional[str],
            required_access_level: AccessLevel,
            required_user_role: Optional[UserRole],
            validation_object_type: token_validation.ValidationObjectType
    ) -> dict:
        if id_token is not None:
            if required_user_role is None:
                raise ValidationFailedError('Id Token authorization is not allowed')

            if self._id_token_verifier is None:
                raise ValidationFailedError('Token Manager lacks an Id Token Verifier')

            if self._user_access_provider is None:
                raise ValidationFailedError('Token Manager lacks a User Access Provider')

            uid = self._id_token_verifier.verify_get_uid(id_token)
            if not uid:
                raise ValidationFailedError('Could not verify id token')

            user_authorized = token_validation.authorize_user(
                project_id,  # type: ignore[arg-type] # TODO: fix type
                scenario_id,  # type: ignore[arg-type] # TODO: fix type
                required_access_level,
                validation_object_type,
                required_user_role,
                uid,
                self._user_access_provider
            )
            if user_authorized:
                return {'user_id': uid}
            else:
                log.debug('Could not authorize id token %s for project %s, scenario %s', id_token, project_id,
                          scenario_id)
                raise ValidationFailedError('Could not authorize id token')

        elif access_token is not None:
            validated, claims = self._validate_access_token(
                access_token,
                prefix,
                project_id,
                scenario_id,
                required_access_level,
                validation_object_type
            )

            if not validated:
                log.debug('Could not authorize access token %s for project %s, scenario %s', access_token, project_id,
                          scenario_id)
                raise ValidationFailedError('Could not authorize access token')

            return claims
        else:
            raise MissingTokenError()

    def generate_access_token(self, refresh_token: str, instance_id: Optional[str] = None) -> str:  # noqa: F841
        refresh_token_data = self.decode_token(refresh_token)
        token_validation.validate_refresh_token(refresh_token_data)

        return self.refresh_access_token({
            'admin': refresh_token_data.get('admin', False),
            **refresh_token_data
        })

    def refresh_access_token(self, refresh_token_data: dict) -> str:
        refresh_token_data.update({
            'type': TokenType.ACCESS,
            'exp': datetime.datetime.utcnow() + ACCESS_TOKEN_EXP
        })

        return self.encode_token(refresh_token_data)

    def generate_refresh_token(
            self,
            token_request=None,
            user_id: Optional[str] = None,
            user_role: Optional[UserRole] = None,
            claims=None,
            project_id=None,
            scenario_id=None,
            admin=False,
            execution_id=None,
            job_id=None,
            organization_id=None,
            agent=None,
            purpose=None,
            access_level=AccessLevel.READ,
            lifetime=None
    ):
        if token_request is not None and user_role is not None:
            user_access_level = USER_ROLE_TO_ACCESS_LEVEL[user_role]
            claims = ApiClaims(
                project_id=token_request.project_id,
                scenario_id=token_request.scenario_id,
                admin=token_request.admin,
                execution_id=token_request.execution_id,
                job_id=token_request.job_id,
                user_id=user_id,
                organization_id=token_request.organization_id,
                access_level=user_access_level.value if token_request.access_level is None  # TODO: AccessLevel type
                else min(token_request.access_level, user_access_level.value))
        elif claims is None:
            claims = ApiClaims(
                project_id=project_id,
                scenario_id=scenario_id,
                admin=admin,
                execution_id=execution_id,
                job_id=job_id,
                user_id=user_id,
                organization_id=organization_id,
                agent=agent,
                purpose=purpose,
                access_level=access_level)

        claims = {
            'type': TokenType.REFRESH,
            'creation_utc_timestamp': datetime.datetime.utcnow().timestamp(),
            **claims.dict()
        }

        if lifetime is not None:
            claims.update({'exp': datetime.datetime.utcnow() + datetime.timedelta(seconds=lifetime)})
        log.debug('Generate refresh token claims: %s', claims)
        return self.encode_token(claims)

    def _validate_access_token(
            self,
            access_token: str,
            requested_prefix: Optional[str],
            requested_project_id: Optional[str],
            requested_scenario_id: Optional[str],
            required_access_level: AccessLevel,
            validation_object_type: ValidationObjectType
    ) -> Tuple[bool, dict]:
        """Validate access token. Returns True if validated and false if not. Also returns claims in token."""
        access_token_data = self.decode_token(access_token)

        token_project_id = access_token_data['project_id']
        token_scenario_id = access_token_data['scenario_id']
        token_admin = access_token_data.get('admin', False)
        token_access_level = AccessLevel(int(access_token_data.get('access_level', AccessLevel.ANY)))  # TODO: int cast
        token_prefix = access_token_data.get('prefix', None)
        claims = access_token_data

        return token_validation.validate_access_token(
            token_project_id=token_project_id,
            token_scenario_id=token_scenario_id,
            token_admin=token_admin,
            token_access_level=token_access_level,
            token_prefix=token_prefix,
            required_access_level=required_access_level,
            requested_project_id=requested_project_id,
            requested_scenario_id=requested_scenario_id,
            requested_prefix=requested_prefix,
            validation_object_type=validation_object_type
        ), claims

    def is_admin_token(self, token: str) -> bool:
        token_data = self.decode_token(token)
        return bool(token_data.get('admin', False))

    def decode_token(self, token: str) -> dict:
        return decode_token(token, key=self._secret)  # type: ignore[no-any-return]

    def encode_token(self, payload: dict) -> str:
        return encode_token(payload, key=self._secret)  # type: ignore[no-any-return]
