import abc
import logging
from enum import Enum
from typing import Optional

from numerous.tokens import AccessLevel, TokenType, UserRole
from numerous.tokens.exceptions import InvalidTokenTypeError, ValidationFailedError

log = logging.getLogger(__name__)


class ValidationObjectType(int, Enum):
    SCENARIO = 0
    GROUP = 1
    PROJECT = 2
    NONE = 3


class UserAccessProvider(abc.ABC):

    @abc.abstractmethod
    def can_access_project(self, user_id: str, project_id: str) -> bool:
        pass

    @abc.abstractmethod
    def can_access_scenario(self, user_id: str, project_id: str, scenario_id: str) -> bool:
        pass

    @abc.abstractmethod
    def has_user_role(self, user_id: str, user_role: UserRole) -> bool:  # noqa: F841
        pass


def validate_access_token(
        token_project_id: str,
        token_scenario_id: str,
        token_admin: bool,
        token_access_level: AccessLevel,
        required_access_level: AccessLevel,
        requested_project_id: Optional[str],
        requested_scenario_id: Optional[str],
        token_prefix: Optional[str] = None,
        requested_prefix: Optional[str] = None,
        validation_object_type: Optional[ValidationObjectType] = None
):
    if token_prefix != requested_prefix:
        return False

    if token_admin:
        return True

    if token_access_level < required_access_level:
        return False

    if token_access_level >= AccessLevel.DEVELOPER:
        return True

    if validation_object_type is None:
        raise ValidationFailedError("No validation object given!")

    if validation_object_type == ValidationObjectType.SCENARIO.value and requested_scenario_id == token_scenario_id:
        return True

    elif validation_object_type == ValidationObjectType.GROUP.value and requested_project_id == token_project_id:
        return True

    elif validation_object_type == ValidationObjectType.PROJECT.value and requested_project_id == token_project_id:
        return True

    return False


def validate_refresh_token(refresh_token_data) -> None:
    if refresh_token_data.get('type', None) != TokenType.REFRESH:
        raise InvalidTokenTypeError('Refresh token required')


def authorize_user(project_id: str, scenario_id: str, required_access_level: AccessLevel,
                   validation_type: ValidationObjectType, required_user_role: UserRole, uid: str,
                   user_access: UserAccessProvider) -> bool:
    if user_access.has_user_role(uid, UserRole.DEVELOPER):
        pass  # Do not validate access to scenario and project
    elif validation_type == ValidationObjectType.SCENARIO and not user_access.can_access_scenario(uid, project_id,
                                                                                                  scenario_id):
        return False
    elif validation_type == ValidationObjectType.GROUP and not user_access.can_access_project(uid, project_id):
        return False
    elif validation_type == ValidationObjectType.PROJECT and not user_access.can_access_project(uid, project_id):
        return False

    if user_access.has_user_role(uid, required_user_role):
        return True
    else:
        log.debug('User did not have required role %s', required_user_role)
        return False
