import sim_components.generic.items as BC
import sim_components.thermodynamics.LF_Pump_with_Map_P as LF_Pump_with_Map
import sim_components.thermodynamics.Fluid_Flow as FF
import sim_components.thermodynamics.Heat_Transfer as HT
import sim_components.thermodynamics.Heat_Exchanger_Nodes_Mean_p as MHX


import numpy as np


class BoreholeStorage_W_HX_Valves(BC.Subsystem):
    def construct_Subsystem(self, Brine, CW, T0=5, V=0.1, pump_scaling=1, kV=5,  **conf):


        #Create BHS




        volume_names = ['BHS_Inlet', 'HX_Supply', 'Return_In', 'Return_Out','Inlet', 'Outlet']
        volumes = self.add_many(item_class=FF.Volume, item_names=volume_names,
                                        **{'Fluid': Brine, 'T0': T0, 'V': 0.1})


        self.Pump = self.add('Pump', LF_Pump_with_Map.Pump_with_Map, scaling=pump_scaling,
                                    inlet=self.BHS_Inlet,
                                    outlet=self.Return_Out, simple=False,
                                    **{'Pump_Type': conf['pump_type'], 'Set_Speed': 1, 'kP': 0.0001}
        )
        self.HX = self.add('HX', MHX.Heat_Exchanger, 3, 600, CW, Brine, T0,
                           T0, 0, F2_1=True, Config='DB300_2', dual='single')

        self.Pump_HX = self.add('Pump_HX', LF_Pump_with_Map.Pump_with_Map, scaling=pump_scaling,
                             inlet=self.HX_Supply,
                             outlet= self.HX.ports['F1A_In'], simple=False,
                             **{'Pump_Type': conf['pump_type'], 'Set_Speed': 0, 'kP': 0.0001}
                             )

        self.BHS = self.add('BHS', BoreholeStorageExchanger, inlet=self.BHS_Inlet, outlet=self.Return_Out, pump=self.Pump, T0=T0, **conf['BHS'])

        self.Return_Valve = self.add('Return_Valve', FF.Valve_Regulated, kV=5, S_set_in=0, side1=self.Return_In, side2=self.Return_Out)

        self.In_Valve = self.add('In_Valve', FF.Valve_Regulated, kV=5, S_set_in=1, side1=self.Inlet,
                                     side2=self.Return_In)

        self.Out_Valve = self.add('Out_Valve', FF.Valve_Regulated, kV=5, S_set_in=1, side1=self.Return_Out,
                                 side2=self.Outlet)



        pipe_lines = [
            [self.BHS_Inlet, self.HX_Supply, self.Return_In],
            [self.HX.ports['F1A_Out'], self.BHS_Inlet],


        ]

        self.HX_VS_Supply = self.HX.ports['F2_In']
        self.HX_VS_Return = self.HX.ports['F2_Out']

        FF.make_pipe_lines(self, pipe_lines, kV=kV)

    def output_(self, sys_dict):
        sys_dict[self.gp('Energy_Usage')] = sys_dict[self.Pump.gp('Energy')] + sys_dict[self.Pump_HX.gp('Energy')]

class EnergyStorage_System(BC.Subsystem):
    def construct_Subsystem(self, Brine, KB, T0=20, V=0.1, BHS_list=[], kV=5, **conf):

        self.BHS_list=BHS_list
        volume_names = ['KB_Pump', 'KB_Return', 'BHS_Supply', 'BHS_Return', 'KB_Supply', 'Return_Short', 'Supply_Short', 'KB_In', 'KB_Out']
        volumes = self.add_many(item_class=FF.Volume, item_names=volume_names,
                                **{'Fluid': Brine, 'T0': T0, 'V': V})
        self.KB = KB
        self.Supply_Pri = self.KB_Out
        self.Return_Pri = self.KB_Return
        #self.KB.Return = KB.Return

        self.Pump = self.add('Pump_KB', LF_Pump_with_Map.Pump_with_Map, scaling=1,
                             inlet=self.KB_Pump,
                             outlet=self.KB.Supply_Pri, simple=False,
                             **{'Pump_Type': conf['pump_type'], 'Set_Speed': 1, 'kP': 0.0001}
                             )

        self.add('Bypass_KB', FF.ThreeWayValve_Regulated, Brine, kV, V, T0,
                 0, pipe_types=[(FF.Pipe,{}), (FF.Pipe_Check,{}), (FF.Pipe,{})],
                 A=self.KB_Out,
                 B=self.KB_In,
                 C=self.KB_Pump
                 )


        self.add('Bypass', FF.ThreeWayValve_Regulated, Brine, kV, V, T0,
                 0, pipe_types=[(FF.Pipe,{}), (FF.Pipe_Check,{'reverse': True}), (FF.Pipe,{})],
                 A=self.KB_Return,
                 B=self.KB_Supply,
                 C=self.KB_In
        )

        for i, BHS in enumerate(BHS_list):
            setattr(self, 'BHS'+str(i+1), BHS)

            self.add('Pipe_BHS_Supply_BHS' + str(i + 1) + '_Inlet', FF.Pipe, kV=kV, side1=self.BHS_Supply,
                     side2=BHS.Inlet)
            self.add('Pipe_BHS'+str(i + 1)+'_Outlet_BHS' + '_Return', FF.Pipe, kV=kV, side1=BHS.Outlet,
                     side2=self.BHS_Return)



        pipe_lines = [
            [self.KB.Return_Pri, self.KB_In],
            [self.Supply_Short, self.KB_Supply, self.BHS_Supply],
            [self.BHS_Return, self.Return_Short],

            [self.KB_Out, self.Supply_Short, ]

        ]



        FF.make_pipe_lines(self, pipe_lines, kV=kV)
        FF.make_pipe_lines(self, [[self.Return_Short, self.Supply_Short]],kV=kV/10)
        FF.make_pipe_check_lines(self, [ [self.Return_Short, self.KB_Return]], kV=kV)

        self.EM_BHS = self.add('EM_BHS', FF.EnergyMeter, inlet=self.BHS_Supply,
                               outlet=self.BHS_Return, pipe=self.Pipe_KB_Supply_BHS_Supply)

        self.calc_funcs = [self.calc]

    def calc(self):

        T_set = KB.T_set
        Err = T_set - KB_Out.T

    def output_(self, sys_dict):
        BHS_energy_usage = sum([sys_dict[BHS.gp('Energy_Usage')] for BHS in self.BHS_list])
        sys_dict[self.gp('Energy_Usage')] = BHS_energy_usage



class BoreholeStorageExchanger(BC.Subsystem):


    def construct_Subsystem(self, inlet=None, outlet=None, pump=None, N_boreholes=63, L_borehole=250, D_borehole=0.044, C_near=1.46337323e+06, C_far=9.48881525e+07,
                            h_near_far= 8.45488164e+01, h_far_ground=5.38126434e+01, h_brine=4.35713508e+02, TG=5, T0=5):
        C_near = C_near / 63 * N_boreholes / 250 * L_borehole
        h_near_far = h_near_far / 63 * N_boreholes / 250 * L_borehole
        C_far = C_far/63*N_boreholes/250*L_borehole
        h_far_ground = h_far_ground/63*N_boreholes/250*L_borehole
        self.inlet = inlet
        self.outlet = outlet
        self.pump = pump

        self.N_boreholes = N_boreholes
        self.L_borehole = L_borehole
        self.D_borehole = D_borehole

        self.V_brine = (self.D_borehole / 2) ** 2 * np.pi * self.L_borehole * 2 * self.N_boreholes
        self.C_brine = self.V_brine * 4.187 * 1000
        self.h_brine = h_brine/63*N_boreholes/250*L_borehole

        self.F_max = N_boreholes * 0.8
        self.P_max = N_boreholes * 15
        # Add volume near
        # C_near = 6.91e6/60/200*self.N_boreholes*self.L_borehole #kJ/kW
        self.V_near = self.add('V_near', HT.Thermal_Mass, C_near, T0=T0)
        # Add volume far
        # C_far = 901e6/60/200*self.N_boreholes*self.L_borehole #kJ/kW
        self.V_far = self.add('V_far', HT.Thermal_Mass, C_far, T0=T0)
        # Add link conductor near-far
        # h_near_far = 444/60/200*self.N_boreholes*self.L_borehole
        self.L_near_far = self.add('L_near_far', HT.Thermal_Conductor, h_near_far, T0=T0, side1=self.V_near,
                                   side2=self.V_far)
        # Add thermal ground
        # h_far_ground = 40/60/200*self.N_boreholes*self.L_borehole
        self.L_far_ground = self.add('L_far_ground', HT.Thermal_Ground, h_far_ground, T_Ground=TG,
                                     side1=self.V_far)

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'state', 'val': 0},
            # {'name': 'P_max', 'unit': 'kW', 'type': 'param', 'val': N_boreholes*15},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'T_brine_out', 'unit': 'oC', 'type': 'param', 'val': 0},
            {'name': 'V_brine', 'unit': 'm3', 'type': 'param', 'val': self.V_brine},
            {'name': 'C_brine', 'unit': 'J/kg/degC', 'type': 'param', 'val': self.C_brine},
            {'name': 'h_C_brine_near', 'unit': '', 'type': 'param', 'val': self.h_brine / self.C_brine},
             {'name': 'E', 'unit': 'MWh', 'type': 'state', 'val': 0},
             {'name': 'W_MWh', 'unit': 'MWh/W', 'type': 'param', 'val': 1.0 / 1e6 / 3600}

        ]

        self.calc_funcs = [self.calc]

    def calc(self):
        apm_dot=abs(pump.m_dot)
        m_dot = apm_dot*(apm_dot>0)+(apm_dot<=0)*0.01
        expo = -V_brine * 1000 / m_dot * h_C_brine_near
        eexpo = exp(expo)
        T_brine_out = V_near.T - (V_near.T - inlet.T) * eexpo
        P = pump.m_dot * 4187 * (inlet.T - T_brine_out)

        outlet.H_dot = -P
        V_near.H_dot = P/1000
        E_dot = P*W_MWh


class DryCooler_W_HX_Valves(BC.Subsystem):
    def construct_Subsystem(self, Brine, CW, T0=5, V=0.1, pump_scaling=1, kV=5,  **conf):


        #Create BHS




        volume_names = ['BHS_Inlet', 'HX_Supply', 'Return_In', 'Return_Out','Inlet', 'Outlet']
        volumes = self.add_many(item_class=FF.Volume, item_names=volume_names,
                                        **{'Fluid': Brine, 'T0': T0, 'V': 0.1})


        self.Pump = self.add('Pump', LF_Pump_with_Map.Pump_with_Map, scaling=pump_scaling,
                                    inlet=self.BHS_Inlet,
                                    outlet=self.Return_Out, simple=False,
                                    **{'Pump_Type': conf['pump_type'], 'Set_Speed': 1, 'kP': 0.0001}
        )
        self.HX = self.add('HX', MHX.Heat_Exchanger, 3, 600, CW, Brine, T0,
                           T0, 0, F2_1=True, Config='DB300_2', dual='single')

        self.Pump_HX = self.add('Pump_HX', LF_Pump_with_Map.Pump_with_Map, scaling=pump_scaling,
                             inlet=self.HX_Supply,
                             outlet= self.HX.ports['F1A_In'], simple=False,
                             **{'Pump_Type': conf['pump_type'], 'Set_Speed': 0, 'kP': 0.0001}
                             )

        self.BHS = self.add('BHS', DryExchanger, inlet=self.BHS_Inlet, outlet=self.Return_Out, pump=self.Pump, T0=T0, **conf['DryCooler'])

        self.Return_Valve = self.add('Return_Valve', FF.Valve_Regulated, kV=5, S_set_in=0, side1=self.Return_In, side2=self.Return_Out)

        self.In_Valve = self.add('In_Valve', FF.Valve_Regulated, kV=5, S_set_in=1, side1=self.Inlet,
                                     side2=self.Return_In)

        self.Out_Valve = self.add('Out_Valve', FF.Valve_Regulated, kV=5, S_set_in=1, side1=self.Return_Out,
                                 side2=self.Outlet)



        pipe_lines = [
            [self.BHS_Inlet, self.HX_Supply, self.Return_In],
            [self.HX.ports['F1A_Out'], self.BHS_Inlet],


        ]

        self.HX_VS_Supply = self.HX.ports['F2_In']
        self.HX_VS_Return = self.HX.ports['F2_Out']

        FF.make_pipe_lines(self, pipe_lines, kV=kV)

    def output_(self, sys_dict):
        sys_dict[self.gp('Energy_Usage')] = sys_dict[self.Pump.gp('Energy')] + sys_dict[self.Pump_HX.gp('Energy')]

class DryExchanger(BC.Subsystem):


    def construct_Subsystem(self, inlet=None, outlet=None, pump=None, C_near=1000,
                             h_far_ground=100, h_brine=4.35713508e+02, TG=5, T0=5):

        self.inlet = inlet
        self.outlet = outlet
        self.pump = pump

        self.V_brine = 10#(self.D_borehole / 2) ** 2 * np.pi * self.L_borehole * 2 * self.N_boreholes
        self.C_brine = self.V_brine * 4.187 * 1000
        self.h_brine = h_brine

        # Add volume near
        # C_near = 6.91e6/60/200*self.N_boreholes*self.L_borehole #kJ/kW
        self.V_near = self.add('V_near', HT.Thermal_Mass, C_near, T0=T0)

        # Add thermal ground
        # h_far_ground = 40/60/200*self.N_boreholes*self.L_borehole
        self.L_air = self.add('L_air', HT.Thermal_Ground, h_far_ground, T_Ground=TG,
                                     side1=self.V_near)

        self.Variables = [
            {'name': 'P', 'unit': 'kW', 'type': 'state', 'val': 0},
            # {'name': 'P_max', 'unit': 'kW', 'type': 'param', 'val': N_boreholes*15},
            {'name': 'E', 'unit': 'MJ', 'type': 'state', 'val': 0},
            {'name': 'T_brine_out', 'unit': 'oC', 'type': 'param', 'val': 0},
            {'name': 'V_brine', 'unit': 'm3', 'type': 'param', 'val': self.V_brine},
            {'name': 'C_brine', 'unit': 'J/kg/degC', 'type': 'param', 'val': self.C_brine},
            {'name': 'h_C_brine_near', 'unit': '', 'type': 'param', 'val': self.h_brine / self.C_brine},
             {'name': 'E', 'unit': 'MWh', 'type': 'state', 'val': 0},
             {'name': 'W_MWh', 'unit': 'MWh/W', 'type': 'param', 'val': 1.0 / 1e6 / 3600}

        ]

        self.calc_funcs = [self.calc]

    def calc(self):
        apm_dot=abs(pump.m_dot)
        m_dot = apm_dot*(apm_dot>0)+(apm_dot<=0)*0.01
        expo = -V_brine * 1000 / m_dot * h_C_brine_near
        eexpo = exp(expo)
        T_brine_out = V_near.T - (V_near.T - inlet.T) * eexpo
        P = pump.m_dot * 4187 * (inlet.T - T_brine_out)

        outlet.H_dot = -P
        V_near.H_dot = P/1000
        E_dot = P*W_MWh
