sentinel = object()

"""Basic Components Module

This module is the container for the basic components (or building blocks) for the model.
"""
import numpy as np
import collections
import warnings


class Item:
    """Basic class for model items. Any item added to a model should extend class Item.
    """

    def __init__(self, name, path, item_fact, *Args, **key_param):
        """
        Args:
            name (str): The local name of the item
            path (str): The path in the model hierachy from top to current item. Each parent item is separated by / or .
            item_fact (Item_Factory): The Item_Factory creating this item. Is passed so this item can create its own children using item_fact via the "get" method.
            *Args: Positional arguments for the item. What is required by the specific item is not determined here - Args are simply passed along.
            **key_param: Keyword arguments for the item, such as the configuration parameters and possibly other items in case of eg. a link.
        """
        self.name = name
        self.diff_id = ''
        self.Variables = []
        self.calc_funcs = []
        self.calc_funcs_str = []

        # Set the path to name if top-level item
        if path == '':
            self.path = name
        # Else, set path to path with name appended
        else:
            self.path = path + '.' + name

        self.item_fact = item_fact

        # Call the construct_Item to make the item construct itself - pass along Args and keyparam
        self.construct_Item(*Args, **key_param)

        # If the item has a method named get_Diff call it to make the item configure its diff parameters to be used when assembling the model.
        if callable(getattr(self, "get_Diff", None)):
            self.diff = self.get_Diff()

    def get(self, name, ItemType, *Args, **keyword):
        """get method. Use to get a new item from the Item_Factory

        Args:
            name  (str): Name for new item
            ItemType (class): Class for new item
            *Args (tuple): Positional arguments for new item
            **keyword (dict): Keyword arguments, such as configuration or named parameters to pass to new item's construct_Item function.

        Returns:
            Item: The new item obtained from the Item_Factory

        """
        # print(keyword)
        # print(Args)
        return self.item_fact.get_Item(name, ItemType, *Args, **keyword)

    def get_paths(self, param):
        """Method for getting full path for a list of parameters

        Args:
            param  (list of str): The parameters of which to return the path

        Returns:
            list of str: A list of paths in the model to the parameters of the current item

        """
        return [self.path + '.' + p for p in param]

    def get_path(self, param):
        # Same as above for one path
        return self.path + '.' + param

    def gp(self, param):
        # Short hand for above
        return self.path + '.' + param

    def get_paths_dict(self, param_dict):
        """Method for getting full path for a dict of parameters

        Args:
            param  (dict): The parameters of which to return the path

        Returns:
            dict: A dict of paths in the model to the parameters of the current item - with the values of param dict.

        """
        return dict(zip([self.path + '.' + p for p in param_dict.keys()], param_dict.values()))

    def remove_itemfact(self):
        delattr(self, 'item_fact')

    def output(self, sys_dict, Final=False):
        """Method for setting output values in the sys_dict. Called by model solver when a time step is successfully integrated and output is generated in form of printing, storing, plotting etc. Meant to be overwritten by classes inheriting Item to add generic behavior for this class.

        Args:
            sys_dict  (dict): A dict with all the variables in the model and their value.

        Returns:
            Nothing

        """
        pass

    def output_(self, sys_dict):
        """Method for setting output values in the sys_dict. Called by model solver when a time step is successfully integrated and output is generated in form of printing, storing, plotting etc. Meant to be overwritten by classes inheriting Item to add specific behavior for an item instance.

        Args:
            sys_dict  (dict): A dict with all the variables in the model and their value.

        Returns:
            Nothing

        """
        pass

    def bind_keys(self, dict, keys):
        for key in keys:
            setattr(self, key, dict[key])

    def add_var(self, name, conf={}, unit='1', vartype='param', val=0, desc='', bind=False):
        val_ = conf[name] if name in conf else val
        self.Variables.append({'name': name, 'unit': unit, 'type': vartype, 'val': val_, 'description': desc})
        if bind:
            setattr(self, name, val_)

    def add_calc_func_str(self, name, code_str):
        self.calc_funcs_str.append({'name': name, 'source': code_str})

    def add_simple_forward_map(self, forward_to_list, map):
        for forward_to in forward_to_list:
            forward_assigns = []
            for k, v in map.items():
                forward_assigns.append('{}.{} = {}'.format(forward_to.name, v, k).strip())

            forward_str = '\n'.join(forward_assigns)
            print('adding forward code:\n', forward_str)
            self.add_calc_func_str('forward_' + self.name + '_' + forward_to.name, forward_str)


class Item_Factory:
    """Class for making model Items and returning them to the caller. The Item_Factory gets the configuration specified and passes it to the Item constructor which returns the new item which is then passed to the caller.
    """

    def __init__(self, name):
        """
        Args:
            name (str): Descriptive name of the Item_Factory
        """
        self.name = name
        self.pos = (0, 0)

    def getConfig(self, config):
        print('Gettig config: ' + config)
        print('Description: ' + self.configs[config].desc + ' - Comments: ' + self.configs[config].comm)
        return self.configs[config]

    def get_Item(self, name, parent_path, anItem, *Args, **key_param):
        attr = {}

        return anItem(name, parent_path, self, *Args, **key_param)

    def store_Item(self, name, desc, comm, conf):
        print('Storing config: ' + name)
        self.configs.update({name: Config(desc, comm, conf)})

    def set_pos(self, offset, scale=(1, 1)):
        self.pos = (self.pos[0] / scale[0] + offset[0], self.pos[1] / scale[1] + offset[1])


class Config:
    def __init__(self, desc, comm, config_items):
        self.config = config_items

        self.desc = desc
        self.comm = comm


class Node(Item):
    def construct_Item(self, *Args, **kwargs):
        self.construct_Node(*Args, **kwargs)
        # self.states=self.create_States_w_Initial_Conditions()

        # self.linked_nodes=[]
        # for k,v in kwargs.items():
        #     if isinstance(v,Item):
        #         self.linked_nodes+=[v]

        # for v in Args:
        #     if isinstance(v,Item):
        #         self.linked_nodes+=[v]


class Link(Item):
    def construct_Item(self, *Args, **key_param):
        self.construct_Link(*Args, **key_param)

        self.linked_nodes = []
        for k, v in key_param.items():
            if isinstance(v, Item):
                self.linked_nodes += [v]

        # for v in Args:
        #     if isinstance(v,Item):
        #         self.linked_nodes+=[v]


class Subsystem(Node):
    def construct_Node(self, *Args, **kwargs):
        self.items = collections.OrderedDict({})
        self.children = []
        self.ports = {}
        try:
            self.construct_Subsystem(*Args, **kwargs)
        except TypeError as te:
            print('item name: ', self.name)
            raise

        # for l, p in self.layout.items():
        #     self.items[self.gp(l)].set_pos(p)

    def add(self, name, ItemType, *Args, **keywords):
        item = self.get(name, self.path, ItemType, *Args, **keywords)
        self.items.update({name: item})
        # print(self.path)
        self.children += [item]
        setattr(self, name, item)
        if hasattr(item, 'items'):
            self.items.update(item.get_paths_dict(item.items))
            # print('')
            # print(self.name)
            # print(self.items.keys())
            # print('')
        return item

    def add_many(self, item_class, item_names, *a, **c):
        new_items = {}
        for name in item_names:
            new_items[name] = self.add(name, item_class, *a, **c)

        return new_items

    def rgetattr(self, obj, attrarr, default=sentinel):
        import functools

        if default is sentinel:
            _getattr = getattr
        else:
            def _getattr(obj, name):
                return getattr(obj, name, default)
        return functools.reduce(_getattr, [obj] + attrarr)

    def get_item_name(self, name, recursive=True):
        namearr = name.split('.')
        obj = self.items[namearr[0]]
        if len(namearr) > 1:
            return self.rgetattr(obj, namearr[1:])
        else:
            return obj

        # return self.items['name']

        # raise ValueError('Item '+self.name+' does not have item with name "'+name+'".')

    def addPort(self, port_name, port_node):
        self.ports.update({port_name: port_node})

    def addPorts(self, new_ports):
        self.ports.update(new_ports)

    def output(self, sys_dict):
        for i in self.items.values():
            i.output(sys_dict)

        return self.output_(sys_dict)

    def output_(self, sys_dict):
        pass

    def set_pos(self, offset, scale=(1, 1)):
        for i in self.items.values():
            i.set_pos(offset, scale)

    def set_outputs(self, output_fields, sys_dict, Final=False):

        outputs = [0 for of in output_fields]
        return outputs

    def set_results(self, output_fields, sys_dict, Final=False):
        return self.set_outputs(output_fields, sys_dict, Final=True)


def rgetattr(obj, attr, default=sentinel):
    import functools

    if default is sentinel:
        _getattr = getattr
    else:
        def _getattr(obj, name):
            return getattr(obj, name, default)
    return functools.reduce(_getattr, [obj] + attr.split('.'))


def exchange_string_for_attr_list(self, the_list):
    for i, l in enumerate(the_list):
        if type(l) == list:
            exchange_string_for_attr_list(self, l)
        elif type(l) == dict:
            exchange_string_for_attr_dict(self, l)
        else:
            try:
                if l[0] == '@':
                    item_name = l[1:]
                    # the_list[i] = getattr(self, item_name)
                    try:
                        the_list[i] = rgetattr(self, item_name)
                    except AttributeError as ae:
                        print(f'System does not have ´{item_name}´, will set argument to None!')
                        warnings.WarningMessage(f'System does not have ´{item_name}´, will set argument to None!')
                        the_list[i] = None


            except TypeError as te:
                pass
            except KeyError as ke:
                pass
            except IndexError as ie:
                pass
            except:
                print('item: ', i)
                print(l)
                raise


def exchange_string_for_attr_dict(self, the_dict):
    for k, v in the_dict.items():
        if type(v) == list:
            exchange_string_for_attr_list(self, v)
        elif type(v) == dict:
            exchange_string_for_attr_dict(self, v)
        else:
            try:
                # print('v: ', v)
                if isinstance(v, str) and len(v) > 0  and v[0] == '@':
                    item_name = v[1:]
                    # the_dict[k] = getattr(self, item_name)
                    # the_dict[k] = rgetattr(self, item_name)
                    try:
                        the_dict[k] = rgetattr(self, item_name)
                    except AttributeError as ae:
                        print(f'System does not have ´{item_name}´, will set argument to None!')
                        #warnings.WarningMessage(f'System does not have ´{item_name}´, will set argument to None!')
                        the_dict[k] = None

            #except TypeError as te:
                # print(te)
            #    pass
            #except KeyError as ke:
                # print(ke)
            #    pass

            #except IndexError as ie:
                # print(ie)
            #    pass

            except:
                print('item: ', k)
                print(v)
                print(the_dict)
                raise


class SubsystemPrescribed(Subsystem):
    def construct_Subsystem(self, specification, *args, **kwargs):
        print('specification: ', specification['items'])
        for k, v in specification['items'].items():
            # print('Item: ',k)
            # print('Args: ', v)

            exchange_string_for_attr_dict(self, v)

            print(v)

            for kk, p in v.items():
                if isinstance(p, str) and len(p) > 0 and p[0]=='@':
                    print(k)
                    print(p)

            try:
                disabled = v['__disabled']
                v.pop('__disabled')
                if disabled:
                    setattr(self, k, None)
                else:
                    self.add(k, None, **v)
            except:
                print(v)
                raise