# Code for simple energy system models
import numpy as np
import numbers
import pickle

import sim_components.generic.items as B
import sim_components.thermodynamics.Heat_Transfer as HT

def calculate_solar_angles(phi = 35, lon = 12, lon_ref = 15, month = 1, day = 10, hour = 13, beta = 35, gamma = 0):
    """
                    Nomenclature:
                        month: number of month (1-January, 12 December)
                        hour: (0-23) - must refer to Daylight Saving Time(DST)
                        day: (0-31) must refer to the day of the month
                        h_culm: solar noon - time in DST at which solar noon occurs
                        E: equation of time

                        Angles (all angles expressed in degrees):
                            phi: Latitude (angular location north or south of the equator)
                            lon : longitude (angular location east or west of fundamental meridian)
                            lon_ref : longitude of the reference meridian! NB! change with the location
                            delta: Declination angle - angular position of the sun at solar noon with respect to the plane of the equator (N positive, S negative)
                            beta : Slope, angle between the plane of the surface in question (e.g. PVT panel) and the horizontal. 0<beta<180 (beta>90 -> downward facing surface)
                            gamma: surface azimuth angle: deviation of the projection on a horizontal plane of the normal to the surface from the local meridian
                                (gamma = 0 PVT oriented to south, <0 for east of south, >0 for west of south)
                            omega: hour angle - angular displacement of the sun east or west of the local meridian due to rotation of the earth axis 15 deg per hour
                            theta: angle of incidence: angle between the beam radiation on a surface and the normal to the surface
                            theta_z: zenith angle, angle between the vertical and the line to the sun (incidence angle of a horizontal surface)
                            alpha_s: solar altitude angle: angle between the beam radiation on a surface and the normal to that surface
                            gamma_s: solar azimuth angle: angular displacement from south of the projection of beam radiation on the horizontal plane

                    Reference: Solar Engineering of Thermal Processes, Duffie and Beckman

            """

    # Calculate day of the year
    starting_day = [0,31,59,90,120,151,181,212,243,273,304,334]
    n = day + starting_day[month-1]
    B = (n-1)*360/365

    # Cooper correlation for declination angle 1969 (for latitudes < 66.5)
    delta = 23.45*np.sin(360*(284+n)/365)

    E =229.2*(0.000075 + 0.001868 *np.cos(B*np.pi/180) -  0.032077*np.sin(B*np.pi/180) - 0.014615* np.cos(2*B*np.pi/180) -  0.04089 * np.sin(2*B*np.pi/180))
    h_culm = 12+(lon-lon_ref)/15-E+hour
    omega = (hour-h_culm)*15

    # zenith angle
    cos_theta_z = np.cos(phi*np.pi/180)*np.cos(delta*np.pi/180)*np.cos(omega*np.pi/180) + np.sin(phi*np.pi/180)*np.sin(delta*np.pi/180)
    theta_z = np.arccos(cos_theta_z)*180/np.pi
    # solar azimuth angle
    cos_gamma_s = (cos_theta_z*np.sin(phi*np.pi/180)-np.sin(delta*np.pi/180))/(np.sin(theta_z*np.pi/180)*np.cos(phi*np.pi/180))
    gamma_s = np.sign(omega) * np.arccos(cos_gamma_s)*180/np.pi

    cos_theta = np.cos(theta_z*np.pi/180)*np.cos(beta*np.pi/180)+np.sin(theta_z*np.pi/180)*np.sin(beta*np.pi/180)*np.cos((gamma_s-gamma)*np.pi/180)
    theta = np.arccos(cos_theta)*180/np.pi

    Rb = cos_theta/cos_theta_z

    return Rb, cos_theta_z


class PVT_Module_Simple(B.Subsystem):
    def construct_Subsystem(self, N=1, A_module = 1.635, Mass_Module = 25,
                            eta_el_0 = 0.183, eta_th_0 = 0.589,
                            NOCT = 45, eta_el_coeff = -0.39, a_1 = 16, a_2 = 0,
                            beta = 30, gamma = 0,
                            phi = 35, lon = 12, lon_ref = 15,
                            mode_el = 1,
                            epsilon_cover = 0.05,
                            **c):

        self.add_var('T_Cell', val=NOCT)
        self.add_var('T_Brine_Out', val=7)
        self.add_var('T_Brine_In', val=7)
        self.add_var('Cp_W', val=4187)
        self.add_var('F_Brine', val=0.02*N) #kg/s

        self.add_var('mode_el', val=mode_el)
        self.add_var('E_th', vartype='state')
        self.add_var('E_el', vartype='state')
        self.add_var('P_th', val=0)
        self.add_var('P_el', val=0)
        self.add_var('Q_losses', val=0)

        self.add_var('C_Cell', val=Mass_Module*N*800)
        self.add_var('N', val=N) #number of PVT modules
        self.add_var('A_module', val=A_module)
        self.add_var('Mass_Module', val=Mass_Module)
        self.add_var('eta_el_0', val=eta_el_0)
        self.add_var('eta_th_0', val=eta_th_0)
        self.add_var('NOCT', val=NOCT)
        self.add_var('eta_el_coeff', val=eta_el_coeff)
        self.add_var('a_1', val=a_1)
        self.add_var('a_2', val=a_2)
        self.add_var('beta', val=beta)
        self.add_var('gamma', val=gamma)
        self.add_var('phi', val=phi) #latitude
        self.add_var('lon', val=lon)  # longitude
        self.add_var('N', val=N)  # number of PVT modules
        self.add_var('epsilon_cover', val=epsilon_cover)  # number of PVT modules

        self.add_var('sigma_B', val=5.67*10**(-8))  # number of PVT modules

        [self.Rb,self.cos_theta_z] = calculate_solar_angles(phi = phi, lon = lon, lon_ref = lon_ref, month = month,
                                    day = Iday, hour = hour, beta = beta, gamma = gamma)
        # Add volume

        self.PV_Mass = self.add('PV_Mass', HT.Thermal_Mass, C = C_Cell, T0=T_Ambient)

        self.calc_funcs = []
        self.calc_funcs.append(self.calc_PVT)

    def calc_PVT(self):
        # the input data must contain Ib, Id and T amb
        T_Cell = PV_Mass.T

        # no need to convert Wh in J as they are already hourly values
        S = Rb*Ib_normal*cos_theta_z + Id*((1+cos(beta* 3.14159 / 180))/2)
        eta_el = eta_el_0*(1-eta_el_coeff/100*(T_Cell-NOCT))

        eta_th = eta_th_0+a_1*T_red+a_2*T_red**2*S

        T_red = (T_m - T_Ambient)/S

        T_m = (T_Brine_In+T_Brine_Out)/2
        T_Brine_Out = T_Brine_In+P_th/Cp_W/F_Brine

        P_el = eta_el*S*A_module*N*mode_el
        P_th = eta_th*S*A_module*N

        # Radiation plus convection
        T_sky = T_Ambient -15
        U_G = 5.7+3.8*Wind_Speed
        Q_losses = (epsilon_cover*sigma_B *(T_Cell**4-T_sky**4) + U_G*((T_Cell**2-T_Ambient**2)))*A_module*N

        PV_Mass.H_dot = S - P_el - P_th - Q_losses
        E_el_dot = P_el* J_MWh
        E_th_dot = P_th* J_MWh


class PVT_System(B.Subsystem):
    def construct_Subsystem(self, PVT=None):
        self.add('PVT', PVT_Module_Simple, **PVT)

if __name__ == '__main__':
    [output1, output2] = calculate_solar_angles(phi = 35, lon = 12, lon_ref = 15, month = 2, day = 10, hour = 0, beta = 35, gamma = 0)
    print(output1)
    print(output2)
